import React, { useEffect, useState } from 'react';
import { useNavigate } from 'react-router-dom';
import api from '@/services/api';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { ShoppingCart, Trash2, Plus, Minus, Package } from 'lucide-react';

const Cart = () => {
  const [cart, setCart] = useState([]);
  const [client, setClient] = useState(null);
  const [loading, setLoading] = useState(false);
  const navigate = useNavigate();

  useEffect(() => {
    loadCart();
    fetchClient();
  }, []);

  const loadCart = () => {
    const savedCart = localStorage.getItem('cart');
    if (savedCart) {
      setCart(JSON.parse(savedCart));
    }
  };

  const fetchClient = async () => {
    try {
      // Essayer de récupérer l'utilisateur actuel
      const userResponse = await api.get('/auth/me');
      const currentUser = userResponse.data;
      
      // Si l'utilisateur est un client, récupérer ses données client
      if (currentUser.role === 'client') {
        // Chercher le client par email
        const clientResponse = await api.get(`/clients?search=${currentUser.email}`);
        if (clientResponse.data.length > 0) {
          setClient(clientResponse.data[0]);
        } else {
          // Si pas de client trouvé, en créer un automatiquement
          const newClient = {
            company_name: currentUser.full_name,
            contact_name: currentUser.full_name,
            email: currentUser.email,
            phone: currentUser.phone || '',
            address: '',
            city: '',
            category: 'garage',
            credit_limit: 100000
          };
          const createResponse = await api.post('/clients', newClient);
          setClient(createResponse.data);
        }
      }
    } catch (error) {
      console.error('Error fetching client:', error);
    }
  };

  const saveCart = (newCart) => {
    setCart(newCart);
    localStorage.setItem('cart', JSON.stringify(newCart));
  };

  const updateQuantity = (productId, change) => {
    const newCart = cart.map(item => {
      if (item.id === productId) {
        const newQuantity = item.quantity + change;
        return newQuantity > 0 ? { ...item, quantity: newQuantity } : null;
      }
      return item;
    }).filter(Boolean);
    
    saveCart(newCart);
  };

  const removeFromCart = (productId) => {
    const newCart = cart.filter(item => item.id !== productId);
    saveCart(newCart);
  };

  const clearCart = () => {
    setCart([]);
    localStorage.removeItem('cart');
  };

  const handleCheckout = async () => {
    if (!client) {
      alert('Profil client non trouvé');
      return;
    }

    setLoading(true);
    try {
      const items = cart.map(item => ({
        product_id: item.id,
        product_name: item.name,
        product_reference: item.reference,
        quantity: item.quantity,
        unit_price: item.prices.garage,
        subtotal: item.prices.garage * item.quantity
      }));

      await api.post('/orders', {
        client_id: client.id,
        items: items,
      });

      clearCart();
      navigate('/client/orders');
    } catch (error) {
      console.error('Error creating order:', error);
      alert('Erreur lors de la création de la commande');
    } finally {
      setLoading(false);
    }
  };

  const subtotal = cart.reduce((sum, item) => sum + (item.prices.garage * item.quantity), 0);
  const taxAmount = subtotal * 0.19;
  const total = subtotal + taxAmount;

  return (
    <div className="space-y-6" data-testid="cart-page">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold text-gray-900">Mon Panier</h1>
          <p className="text-gray-600 mt-2">{cart.length} article(s) dans votre panier</p>
        </div>
        <Button 
          onClick={() => navigate('/client/catalog')}
          variant="outline"
          data-testid="continue-shopping-button"
        >
          <Package className="h-4 w-4 mr-2" />
          Continuer mes achats
        </Button>
      </div>

      {cart.length === 0 ? (
        <Card>
          <CardContent className="py-16 text-center">
            <ShoppingCart className="h-16 w-16 mx-auto text-gray-400 mb-4" />
            <h3 className="text-xl font-semibold text-gray-900 mb-2">Votre panier est vide</h3>
            <p className="text-gray-600 mb-6">Parcourez notre catalogue pour ajouter des produits</p>
            <Button 
              onClick={() => navigate('/client/catalog')}
              className="bg-indigo-600 hover:bg-indigo-700"
            >
              Voir le catalogue
            </Button>
          </CardContent>
        </Card>
      ) : (
        <div className="grid lg:grid-cols-3 gap-6">
          {/* Cart Items */}
          <div className="lg:col-span-2 space-y-4">
            <Card>
              <CardHeader>
                <CardTitle>Articles</CardTitle>
                <CardDescription>Vérifiez et modifiez votre commande</CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                {cart.map((item) => (
                  <div key={item.id} className="flex items-center gap-4 p-4 bg-gray-50 rounded-lg" data-testid={`cart-item-${item.id}`}>
                    {item.images && item.images.length > 0 && (
                      <img 
                        src={item.images[0]} 
                        alt={item.name}
                        className="w-20 h-20 object-cover rounded-lg bg-white"
                      />
                    )}
                    
                    <div className="flex-1">
                      <h4 className="font-semibold">{item.name}</h4>
                      <p className="text-sm text-gray-600">Réf: {item.reference}</p>
                      <p className="text-lg font-bold text-indigo-600 mt-1">
                        {new Intl.NumberFormat('fr-DZ', {
                          style: 'currency',
                          currency: 'DZD',
                        }).format(item.prices.garage)}
                      </p>
                    </div>

                    <div className="flex items-center space-x-2">
                      <Button
                        variant="outline"
                        size="sm"
                        onClick={() => updateQuantity(item.id, -1)}
                        data-testid={`decrease-${item.id}`}
                      >
                        <Minus className="h-4 w-4" />
                      </Button>
                      <span className="w-12 text-center font-bold">{item.quantity}</span>
                      <Button
                        variant="outline"
                        size="sm"
                        onClick={() => updateQuantity(item.id, 1)}
                        disabled={item.quantity >= item.stock_qty}
                        data-testid={`increase-${item.id}`}
                      >
                        <Plus className="h-4 w-4" />
                      </Button>
                    </div>

                    <div className="text-right">
                      <p className="font-bold text-lg">
                        {new Intl.NumberFormat('fr-DZ', {
                          style: 'currency',
                          currency: 'DZD',
                        }).format(item.prices.garage * item.quantity)}
                      </p>
                      <Button
                        variant="ghost"
                        size="sm"
                        className="text-red-600 hover:text-red-700 mt-2"
                        onClick={() => removeFromCart(item.id)}
                        data-testid={`remove-${item.id}`}
                      >
                        <Trash2 className="h-4 w-4" />
                      </Button>
                    </div>
                  </div>
                ))}
              </CardContent>
            </Card>
          </div>

          {/* Order Summary */}
          <div>
            <Card className="sticky top-24">
              <CardHeader>
                <CardTitle>Récapitulatif</CardTitle>
                <CardDescription>Détails de votre commande</CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="space-y-2">
                  <div className="flex justify-between">
                    <span className="text-gray-600">Sous-total</span>
                    <span className="font-medium">
                      {new Intl.NumberFormat('fr-DZ', {
                        style: 'currency',
                        currency: 'DZD',
                      }).format(subtotal)}
                    </span>
                  </div>
                  <div className="flex justify-between">
                    <span className="text-gray-600">TVA (19%)</span>
                    <span className="font-medium">
                      {new Intl.NumberFormat('fr-DZ', {
                        style: 'currency',
                        currency: 'DZD',
                      }).format(taxAmount)}
                    </span>
                  </div>
                  <div className="border-t pt-2 flex justify-between text-lg font-bold">
                    <span>Total TTC</span>
                    <span className="text-indigo-600">
                      {new Intl.NumberFormat('fr-DZ', {
                        style: 'currency',
                        currency: 'DZD',
                      }).format(total)}
                    </span>
                  </div>
                </div>

                <Button
                  className="w-full bg-indigo-600 hover:bg-indigo-700"
                  onClick={handleCheckout}
                  disabled={loading || !client}
                  data-testid="checkout-button"
                >
                  {loading ? 'Traitement...' : 'Valider la commande'}
                </Button>

                <Button
                  variant="outline"
                  className="w-full"
                  onClick={clearCart}
                  data-testid="clear-cart-button"
                >
                  Vider le panier
                </Button>

                <div className="text-xs text-gray-500 text-center mt-4">
                  <p>✓ Paiement sécurisé</p>
                  <p>✓ Livraison rapide</p>
                  <p>✓ Service après-vente</p>
                </div>
              </CardContent>
            </Card>
          </div>
        </div>
      )}
    </div>
  );
};

export default Cart;
