import React, { useState } from 'react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Badge } from '@/components/ui/badge';
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle, DialogTrigger } from '@/components/ui/dialog';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Wrench, Plus, Calendar, Clock, Car, CheckCircle } from 'lucide-react';

const WorkshopManagement = () => {
  const [repairOrders, setRepairOrders] = useState([
    {
      id: 'OR-001',
      order_number: 'OR-2025-001',
      client_name: 'Mohamed Benali',
      vehicle: 'Renault Clio 2018',
      registration: '16-12345-16',
      vin: 'VF15RD00054123456',
      entry_date: '2025-01-15',
      promised_date: '2025-01-18',
      status: 'in_progress',
      mechanic: 'Karim M.',
      operations: [
        { name: 'Vidange moteur', duration: 30, price: 2500 },
        { name: 'Changement plaquettes frein', duration: 60, price: 8500 },
        { name: 'Contrôle suspension', duration: 20, price: 1500 },
      ],
      parts: [
        { name: 'Filtre à huile', ref: 'FO-001', qty: 1, price: 650 },
        { name: 'Plaquettes frein avant', ref: 'PQ-100', qty: 1, price: 3800 },
        { name: 'Huile moteur 5W40', ref: 'HL-200', qty: 5, price: 4500 },
      ],
      total_labor: 12500,
      total_parts: 8950,
      total: 21450,
    },
    {
      id: 'OR-002',
      order_number: 'OR-2025-002',
      client_name: 'Fatima Saidi',
      vehicle: 'Peugeot 208 2020',
      registration: '16-98765-16',
      vin: 'VF3C5FH0HKS234567',
      entry_date: '2025-01-16',
      promised_date: '2025-01-17',
      status: 'completed',
      mechanic: 'Ahmed B.',
      operations: [
        { name: 'Diagnostic électronique', duration: 45, price: 3500 },
        { name: 'Remplacement batterie', duration: 15, price: 1000 },
      ],
      parts: [
        { name: 'Batterie 70Ah', ref: 'BAT-70', qty: 1, price: 12000 },
      ],
      total_labor: 4500,
      total_parts: 12000,
      total: 16500,
    },
  ]);

  const [showDialog, setShowDialog] = useState(false);
  const [selectedOrder, setSelectedOrder] = useState(null);
  const [showDetailsDialog, setShowDetailsDialog] = useState(false);
  const [formData, setFormData] = useState({
    client_name: '',
    vehicle: '',
    registration: '',
    vin: '',
    promised_date: '',
    mechanic: '',
  });

  const mechanics = ['Karim M.', 'Ahmed B.', 'Yacine K.', 'Sofiane R.'];

  const getStatusBadge = (status) => {
    const config = {
      pending: { variant: 'outline', label: 'En attente' },
      in_progress: { variant: 'info', label: 'En cours' },
      waiting_parts: { variant: 'warning', label: 'Attente pièces' },
      waiting_validation: { variant: 'warning', label: 'Validation client' },
      completed: { variant: 'success', label: 'Terminé' },
      delivered: { variant: 'success', label: 'Livré' },
      cancelled: { variant: 'destructive', label: 'Annulé' },
    };
    const c = config[status] || config.pending;
    return <Badge variant={c.variant}>{c.label}</Badge>;
  };

  const openDetailsDialog = (order) => {
    setSelectedOrder(order);
    setShowDetailsDialog(true);
  };

  const stats = {
    total: repairOrders.length,
    in_progress: repairOrders.filter(o => o.status === 'in_progress').length,
    completed_today: repairOrders.filter(o => o.status === 'completed').length,
    revenue_today: repairOrders.filter(o => o.status === 'completed' || o.status === 'delivered')
      .reduce((sum, o) => sum + o.total, 0),
  };

  return (
    <div className="space-y-6" data-testid="workshop-page">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold text-gray-900">Gestion Atelier</h1>
          <p className="text-gray-600 mt-2">Ordres de réparation et planning mécanique</p>
        </div>
        <Button
          className="bg-indigo-600 hover:bg-indigo-700"
          onClick={() => setShowDialog(true)}
          data-testid="add-repair-order-button"
        >
          <Plus className="h-4 w-4 mr-2" />
          Nouvel OR
        </Button>
      </div>

      {/* Stats Cards */}
      <div className="grid gap-6 md:grid-cols-4">
        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">OR Actifs</CardTitle>
            <Wrench className="h-4 w-4 text-blue-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{stats.total}</div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">En Cours</CardTitle>
            <Clock className="h-4 w-4 text-orange-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-orange-600">{stats.in_progress}</div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Terminés Aujourd'hui</CardTitle>
            <CheckCircle className="h-4 w-4 text-green-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-green-600">{stats.completed_today}</div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">CA Atelier Aujourd'hui</CardTitle>
            <Wrench className="h-4 w-4 text-purple-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-purple-600">
              {new Intl.NumberFormat('fr-DZ', {
                style: 'currency',
                currency: 'DZD',
                minimumFractionDigits: 0,
              }).format(stats.revenue_today)}
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Planning View */}
      <Card>
        <CardHeader>
          <CardTitle>Planning Atelier</CardTitle>
          <CardDescription>Vue d'ensemble des ordres de réparation</CardDescription>
        </CardHeader>
        <CardContent>
          <div className="overflow-x-auto">
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead>N° OR</TableHead>
                  <TableHead>Client</TableHead>
                  <TableHead>Véhicule</TableHead>
                  <TableHead>Immat.</TableHead>
                  <TableHead>Mécanicien</TableHead>
                  <TableHead>Entrée</TableHead>
                  <TableHead>Promis</TableHead>
                  <TableHead className="text-right">Total</TableHead>
                  <TableHead>Statut</TableHead>
                  <TableHead className="text-center">Actions</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {repairOrders.map((order) => (
                  <TableRow key={order.id}>
                    <TableCell className="font-medium">{order.order_number}</TableCell>
                    <TableCell>{order.client_name}</TableCell>
                    <TableCell>
                      <div className="flex items-center gap-2">
                        <Car className="h-4 w-4 text-gray-400" />
                        {order.vehicle}
                      </div>
                    </TableCell>
                    <TableCell className="font-mono text-sm">{order.registration}</TableCell>
                    <TableCell>{order.mechanic}</TableCell>
                    <TableCell>{new Date(order.entry_date).toLocaleDateString('fr-FR')}</TableCell>
                    <TableCell className="font-semibold">
                      {new Date(order.promised_date).toLocaleDateString('fr-FR')}
                    </TableCell>
                    <TableCell className="text-right font-bold text-green-600">
                      {new Intl.NumberFormat('fr-DZ', {
                        style: 'currency',
                        currency: 'DZD',
                      }).format(order.total)}
                    </TableCell>
                    <TableCell>{getStatusBadge(order.status)}</TableCell>
                    <TableCell className="text-center">
                      <Button
                        variant="outline"
                        size="sm"
                        onClick={() => openDetailsDialog(order)}
                        data-testid={`view-order-${order.id}`}
                      >
                        Détails
                      </Button>
                    </TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          </div>
        </CardContent>
      </Card>

      {/* Mechanics Workload */}
      <div className="grid md:grid-cols-4 gap-4">
        {mechanics.map(mechanic => {
          const mechanicOrders = repairOrders.filter(o => o.mechanic === mechanic && o.status === 'in_progress');
          return (
            <Card key={mechanic}>
              <CardHeader>
                <CardTitle className="text-base flex items-center gap-2">
                  <Wrench className="h-4 w-4" />
                  {mechanic}
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-2">
                  <div>
                    <p className="text-sm text-gray-500">En cours</p>
                    <p className="text-2xl font-bold">{mechanicOrders.length}</p>
                  </div>
                  <div>
                    <p className="text-sm text-gray-500">Charge de travail</p>
                    <div className="w-full bg-gray-200 rounded-full h-2 mt-1">
                      <div
                        className={`h-2 rounded-full ${
                          mechanicOrders.length > 2 ? 'bg-red-500' : 'bg-green-500'
                        }`}
                        style={{ width: `${Math.min(mechanicOrders.length * 33, 100)}%` }}
                      />
                    </div>
                  </div>
                </div>
              </CardContent>
            </Card>
          );
        })}
      </div>

      {/* New Repair Order Dialog */}
      <Dialog open={showDialog} onOpenChange={setShowDialog}>
        <DialogContent className="max-w-2xl">
          <DialogHeader>
            <DialogTitle>Nouvel Ordre de Réparation</DialogTitle>
            <DialogDescription>Enregistrer un nouveau véhicule en atelier</DialogDescription>
          </DialogHeader>
          <div className="grid gap-4 py-4">
            <div className="grid grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label>Client</Label>
                <Input
                  value={formData.client_name}
                  onChange={(e) => setFormData({...formData, client_name: e.target.value})}
                  placeholder="Nom du client"
                />
              </div>
              <div className="space-y-2">
                <Label>Véhicule</Label>
                <Input
                  value={formData.vehicle}
                  onChange={(e) => setFormData({...formData, vehicle: e.target.value})}
                  placeholder="Marque Modèle Année"
                />
              </div>
            </div>

            <div className="grid grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label>Immatriculation</Label>
                <Input
                  value={formData.registration}
                  onChange={(e) => setFormData({...formData, registration: e.target.value})}
                  placeholder="16-12345-16"
                />
              </div>
              <div className="space-y-2">
                <Label>VIN (Optionnel)</Label>
                <Input
                  value={formData.vin}
                  onChange={(e) => setFormData({...formData, vin: e.target.value})}
                  placeholder="VF15RD00054123456"
                />
              </div>
            </div>

            <div className="grid grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label>Date Promesse</Label>
                <Input
                  type="date"
                  value={formData.promised_date}
                  onChange={(e) => setFormData({...formData, promised_date: e.target.value})}
                />
              </div>
              <div className="space-y-2">
                <Label>Mécanicien</Label>
                <Select
                  value={formData.mechanic}
                  onValueChange={(value) => setFormData({...formData, mechanic: value})}
                >
                  <SelectTrigger>
                    <SelectValue placeholder="Sélectionner" />
                  </SelectTrigger>
                  <SelectContent>
                    {mechanics.map(m => (
                      <SelectItem key={m} value={m}>{m}</SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
            </div>
          </div>
          <DialogFooter>
            <Button variant="outline" onClick={() => setShowDialog(false)}>Annuler</Button>
            <Button className="bg-indigo-600">Créer OR</Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Order Details Dialog */}
      <Dialog open={showDetailsDialog} onOpenChange={setShowDetailsDialog}>
        <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle>Détails OR - {selectedOrder?.order_number}</DialogTitle>
            <DialogDescription>
              {selectedOrder?.client_name} - {selectedOrder?.vehicle}
            </DialogDescription>
          </DialogHeader>
          {selectedOrder && (
            <div className="space-y-6">
              {/* Vehicle Info */}
              <Card>
                <CardHeader>
                  <CardTitle className="text-base">Informations Véhicule</CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="grid grid-cols-2 gap-4">
                    <div>
                      <p className="text-sm text-gray-500">Immatriculation</p>
                      <p className="font-mono font-bold">{selectedOrder.registration}</p>
                    </div>
                    <div>
                      <p className="text-sm text-gray-500">VIN</p>
                      <p className="font-mono text-sm">{selectedOrder.vin}</p>
                    </div>
                    <div>
                      <p className="text-sm text-gray-500">Mécanicien</p>
                      <p className="font-semibold">{selectedOrder.mechanic}</p>
                    </div>
                    <div>
                      <p className="text-sm text-gray-500">Statut</p>
                      {getStatusBadge(selectedOrder.status)}
                    </div>
                  </div>
                </CardContent>
              </Card>

              {/* Operations */}
              <div>
                <h3 className="font-semibold mb-2">Main d'œuvre</h3>
                <div className="border rounded-lg overflow-hidden">
                  <table className="w-full text-sm">
                    <thead className="bg-gray-50">
                      <tr>
                        <th className="px-4 py-2 text-left">Opération</th>
                        <th className="px-4 py-2 text-center">Durée (min)</th>
                        <th className="px-4 py-2 text-right">Prix</th>
                      </tr>
                    </thead>
                    <tbody>
                      {selectedOrder.operations.map((op, idx) => (
                        <tr key={idx} className="border-t">
                          <td className="px-4 py-2">{op.name}</td>
                          <td className="px-4 py-2 text-center">{op.duration}</td>
                          <td className="px-4 py-2 text-right font-semibold">
                            {new Intl.NumberFormat('fr-DZ', {
                              style: 'currency',
                              currency: 'DZD',
                            }).format(op.price)}
                          </td>
                        </tr>
                      ))}
                      <tr className="border-t bg-gray-50 font-bold">
                        <td className="px-4 py-2" colSpan={2}>Total Main d'œuvre</td>
                        <td className="px-4 py-2 text-right">
                          {new Intl.NumberFormat('fr-DZ', {
                            style: 'currency',
                            currency: 'DZD',
                          }).format(selectedOrder.total_labor)}
                        </td>
                      </tr>
                    </tbody>
                  </table>
                </div>
              </div>

              {/* Parts */}
              <div>
                <h3 className="font-semibold mb-2">Pièces</h3>
                <div className="border rounded-lg overflow-hidden">
                  <table className="w-full text-sm">
                    <thead className="bg-gray-50">
                      <tr>
                        <th className="px-4 py-2 text-left">Désignation</th>
                        <th className="px-4 py-2 text-left">Référence</th>
                        <th className="px-4 py-2 text-center">Qté</th>
                        <th className="px-4 py-2 text-right">Prix Unit.</th>
                        <th className="px-4 py-2 text-right">Total</th>
                      </tr>
                    </thead>
                    <tbody>
                      {selectedOrder.parts.map((part, idx) => (
                        <tr key={idx} className="border-t">
                          <td className="px-4 py-2">{part.name}</td>
                          <td className="px-4 py-2 text-gray-600">{part.ref}</td>
                          <td className="px-4 py-2 text-center">{part.qty}</td>
                          <td className="px-4 py-2 text-right">
                            {new Intl.NumberFormat('fr-DZ', {
                              style: 'currency',
                              currency: 'DZD',
                            }).format(part.price)}
                          </td>
                          <td className="px-4 py-2 text-right font-semibold">
                            {new Intl.NumberFormat('fr-DZ', {
                              style: 'currency',
                              currency: 'DZD',
                            }).format(part.price * part.qty)}
                          </td>
                        </tr>
                      ))}
                      <tr className="border-t bg-gray-50 font-bold">
                        <td className="px-4 py-2" colSpan={4}>Total Pièces</td>
                        <td className="px-4 py-2 text-right">
                          {new Intl.NumberFormat('fr-DZ', {
                            style: 'currency',
                            currency: 'DZD',
                          }).format(selectedOrder.total_parts)}
                        </td>
                      </tr>
                    </tbody>
                  </table>
                </div>
              </div>

              {/* Total */}
              <div className="border-t pt-4">
                <div className="flex justify-between text-2xl font-bold">
                  <span>TOTAL TTC</span>
                  <span className="text-green-600">
                    {new Intl.NumberFormat('fr-DZ', {
                      style: 'currency',
                      currency: 'DZD',
                    }).format(selectedOrder.total)}
                  </span>
                </div>
              </div>

              {/* Status Update */}
              <div className="flex gap-2">
                <Select defaultValue={selectedOrder.status}>
                  <SelectTrigger className="flex-1">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="pending">En attente</SelectItem>
                    <SelectItem value="in_progress">En cours</SelectItem>
                    <SelectItem value="waiting_parts">Attente pièces</SelectItem>
                    <SelectItem value="waiting_validation">Validation client</SelectItem>
                    <SelectItem value="completed">Terminé</SelectItem>
                    <SelectItem value="delivered">Livré</SelectItem>
                  </SelectContent>
                </Select>
                <Button>Mettre à jour</Button>
              </div>
            </div>
          )}
          <DialogFooter>
            <Button variant="outline" onClick={() => setShowDetailsDialog(false)}>Fermer</Button>
            <Button className="bg-green-600">Imprimer Facture</Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
};

export default WorkshopManagement;
