import React, { useEffect, useState } from 'react';
import api from '@/services/api';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Badge } from '@/components/ui/badge';
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle } from '@/components/ui/dialog';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import PermissionsSelector from '@/components/PermissionsSelector';
import { Plus, Edit, Trash2, Users, Shield, CheckCircle, XCircle, Key } from 'lucide-react';

const UsersPage = () => {
  const [users, setUsers] = useState([]);
  const [templates, setTemplates] = useState({});
  const [loading, setLoading] = useState(true);
  const [showDialog, setShowDialog] = useState(false);
  const [showPermissionsDialog, setShowPermissionsDialog] = useState(false);
  const [editMode, setEditMode] = useState(false);
  const [selectedUser, setSelectedUser] = useState(null);
  const [selectedTemplate, setSelectedTemplate] = useState('');
  
  const [formData, setFormData] = useState({
    email: '',
    password: '',
    full_name: '',
    phone: '',
    role: 'manager',
    language: 'fr',
    is_active: true,
    permissions: null
  });

  const [permissionsData, setPermissionsData] = useState(null);

  useEffect(() => {
    fetchUsers();
    fetchTemplates();
  }, []);

  const fetchUsers = async () => {
    try {
      const response = await api.get('/users');
      setUsers(response.data);
    } catch (error) {
      console.error('Error fetching users:', error);
    } finally {
      setLoading(false);
    }
  };

  const fetchTemplates = async () => {
    try {
      const response = await api.get('/users/templates/permissions');
      setTemplates(response.data);
    } catch (error) {
      console.error('Error fetching templates:', error);
    }
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    try {
      const userData = {
        ...formData,
        permissions: permissionsData
      };

      if (editMode) {
        await api.put(`/users/${selectedUser.id}`, userData);
        alert('Utilisateur modifié avec succès');
      } else {
        await api.post('/auth/register', userData);
        alert('Utilisateur créé avec succès');
      }
      
      fetchUsers();
      setShowDialog(false);
      resetForm();
    } catch (error) {
      console.error('Error saving user:', error);
      alert(error.response?.data?.detail || 'Erreur lors de la sauvegarde');
    }
  };

  const handleEdit = (user) => {
    setSelectedUser(user);
    setFormData({
      email: user.email,
      password: '',
      full_name: user.full_name,
      phone: user.phone || '',
      role: user.role,
      language: user.language,
      is_active: user.is_active,
      permissions: user.permissions
    });
    setPermissionsData(user.permissions);
    setEditMode(true);
    setShowDialog(true);
  };

  const handleEditPermissions = (user) => {
    setSelectedUser(user);
    setPermissionsData(user.permissions);
    setShowPermissionsDialog(true);
  };

  const handleSavePermissions = async () => {
    try {
      await api.patch(`/users/${selectedUser.id}/permissions`, permissionsData);
      alert('Permissions mises à jour avec succès');
      setShowPermissionsDialog(false);
      fetchUsers();
    } catch (error) {
      console.error('Error updating permissions:', error);
      alert(error.response?.data?.detail || 'Erreur lors de la mise à jour');
    }
  };

  const handleToggleStatus = async (user) => {
    if (!window.confirm(`${user.is_active ? 'Désactiver' : 'Activer'} ${user.full_name} ?`)) {
      return;
    }
    
    try {
      await api.patch(`/users/${user.id}/status`, { is_active: !user.is_active });
      fetchUsers();
      alert(`Utilisateur ${user.is_active ? 'désactivé' : 'activé'} avec succès`);
    } catch (error) {
      console.error('Error toggling status:', error);
      alert(error.response?.data?.detail || 'Erreur lors du changement de statut');
    }
  };

  const handleDelete = async (user) => {
    if (!window.confirm(`Êtes-vous sûr de vouloir supprimer ${user.full_name} ? Cette action est irréversible.`)) {
      return;
    }
    
    try {
      await api.delete(`/users/${user.id}`);
      fetchUsers();
      alert('Utilisateur supprimé avec succès');
    } catch (error) {
      console.error('Error deleting user:', error);
      alert(error.response?.data?.detail || 'Erreur lors de la suppression');
    }
  };

  const handleApplyTemplate = (templateKey) => {
    if (templateKey && templates[templateKey]) {
      setPermissionsData(templates[templateKey].permissions);
      setSelectedTemplate(templateKey);
    }
  };

  const resetForm = () => {
    setFormData({
      email: '',
      password: '',
      full_name: '',
      phone: '',
      role: 'manager',
      language: 'fr',
      is_active: true,
      permissions: null
    });
    setPermissionsData(null);
    setSelectedUser(null);
    setEditMode(false);
    setSelectedTemplate('');
  };

  const getRoleBadge = (role) => {
    const badges = {
      superadmin: { label: 'Super Admin', class: 'bg-red-100 text-red-800' },
      admin: { label: 'Admin', class: 'bg-purple-100 text-purple-800' },
      manager: { label: 'Manager', class: 'bg-blue-100 text-blue-800' },
      client: { label: 'Client', class: 'bg-gray-100 text-gray-800' }
    };
    const badge = badges[role] || badges.client;
    return <Badge className={badge.class}>{badge.label}</Badge>;
  };

  const getPermissionsSummary = (user) => {
    if (!user.permissions) {
      return <span className="text-sm text-gray-500 italic">Par défaut ({user.role})</span>;
    }
    
    // Compter le nombre de modules avec au moins une permission
    const modulesWithPerms = Object.values(user.permissions).filter(
      perms => perms.view || perms.create || perms.edit || perms.delete || perms.approve
    ).length;
    
    return <span className="text-sm text-blue-600 font-medium">{modulesWithPerms} modules personnalisés</span>;
  };

  return (
    <div className="container mx-auto py-8 px-4">
      <Card>
        <CardHeader>
          <div className="flex items-center justify-between">
            <div>
              <CardTitle className="text-2xl flex items-center gap-2">
                <Users className="h-6 w-6" />
                Gestion des Utilisateurs
              </CardTitle>
              <CardDescription className="mt-2">
                Créer et gérer les utilisateurs avec leurs permissions (Réservé au Superviseur)
              </CardDescription>
            </div>
            <Button 
              onClick={() => {
                resetForm();
                setShowDialog(true);
              }}
              className="bg-indigo-600 hover:bg-indigo-700"
            >
              <Plus className="h-4 w-4 mr-2" />
              Nouvel Utilisateur
            </Button>
          </div>
        </CardHeader>
        <CardContent>
          {loading ? (
            <div className="text-center py-8">Chargement...</div>
          ) : users.length === 0 ? (
            <div className="text-center py-8 text-gray-500">Aucun utilisateur</div>
          ) : (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead>Nom Complet</TableHead>
                  <TableHead>Email</TableHead>
                  <TableHead>Téléphone</TableHead>
                  <TableHead>Rôle</TableHead>
                  <TableHead>Permissions</TableHead>
                  <TableHead>Statut</TableHead>
                  <TableHead className="text-right">Actions</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {users.map((user) => (
                  <TableRow key={user.id}>
                    <TableCell className="font-medium">{user.full_name}</TableCell>
                    <TableCell>{user.email}</TableCell>
                    <TableCell>{user.phone || '-'}</TableCell>
                    <TableCell>{getRoleBadge(user.role)}</TableCell>
                    <TableCell>{getPermissionsSummary(user)}</TableCell>
                    <TableCell>
                      {user.is_active ? (
                        <Badge className="bg-green-100 text-green-800">
                          <CheckCircle className="h-3 w-3 mr-1" />
                          Actif
                        </Badge>
                      ) : (
                        <Badge className="bg-red-100 text-red-800">
                          <XCircle className="h-3 w-3 mr-1" />
                          Inactif
                        </Badge>
                      )}
                    </TableCell>
                    <TableCell className="text-right">
                      <div className="flex gap-2 justify-end">
                        <Button
                          size="sm"
                          variant="outline"
                          onClick={() => handleEdit(user)}
                          title="Modifier"
                        >
                          <Edit className="h-4 w-4" />
                        </Button>
                        <Button
                          size="sm"
                          variant="outline"
                          onClick={() => handleEditPermissions(user)}
                          title="Gérer les permissions"
                          className="text-blue-600"
                        >
                          <Key className="h-4 w-4" />
                        </Button>
                        <Button
                          size="sm"
                          variant="outline"
                          onClick={() => handleToggleStatus(user)}
                          title={user.is_active ? 'Désactiver' : 'Activer'}
                          className={user.is_active ? 'text-orange-600' : 'text-green-600'}
                        >
                          {user.is_active ? <XCircle className="h-4 w-4" /> : <CheckCircle className="h-4 w-4" />}
                        </Button>
                        {user.role !== 'superadmin' && (
                          <Button
                            size="sm"
                            variant="outline"
                            onClick={() => handleDelete(user)}
                            title="Supprimer"
                            className="text-red-600"
                          >
                            <Trash2 className="h-4 w-4" />
                          </Button>
                        )}
                      </div>
                    </TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          )}
        </CardContent>
      </Card>

      {/* Create/Edit User Dialog */}
      <Dialog open={showDialog} onOpenChange={setShowDialog}>
        <DialogContent className="max-w-5xl max-h-[90vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle>{editMode ? 'Modifier Utilisateur' : 'Nouvel Utilisateur'}</DialogTitle>
            <DialogDescription>
              {editMode 
                ? 'Modifier les informations de l\'utilisateur' 
                : 'Créer un nouvel utilisateur et définir ses permissions'}
            </DialogDescription>
          </DialogHeader>
          
          <form onSubmit={handleSubmit}>
            <Tabs defaultValue="info" className="w-full">
              <TabsList className="grid w-full grid-cols-2">
                <TabsTrigger value="info">Informations</TabsTrigger>
                <TabsTrigger value="permissions">Permissions</TabsTrigger>
              </TabsList>

              <TabsContent value="info" className="space-y-4 mt-4">
                <div className="grid grid-cols-2 gap-4">
                  <div className="space-y-2">
                    <Label htmlFor="full_name">Nom Complet *</Label>
                    <Input
                      id="full_name"
                      value={formData.full_name}
                      onChange={(e) => setFormData({...formData, full_name: e.target.value})}
                      required
                    />
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="email">Email *</Label>
                    <Input
                      id="email"
                      type="email"
                      value={formData.email}
                      onChange={(e) => setFormData({...formData, email: e.target.value})}
                      required
                      disabled={editMode}
                    />
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="password">Mot de passe {!editMode && '*'}</Label>
                    <Input
                      id="password"
                      type="password"
                      value={formData.password}
                      onChange={(e) => setFormData({...formData, password: e.target.value})}
                      required={!editMode}
                      placeholder={editMode ? 'Laisser vide pour ne pas changer' : ''}
                    />
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="phone">Téléphone</Label>
                    <Input
                      id="phone"
                      value={formData.phone}
                      onChange={(e) => setFormData({...formData, phone: e.target.value})}
                      placeholder="05XX XX XX XX"
                    />
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="role">Rôle *</Label>
                    <Select
                      value={formData.role}
                      onValueChange={(value) => setFormData({...formData, role: value})}
                    >
                      <SelectTrigger>
                        <SelectValue />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="superadmin">Super Admin (Tous droits)</SelectItem>
                        <SelectItem value="admin">Admin (Large accès)</SelectItem>
                        <SelectItem value="manager">Manager (Accès modéré)</SelectItem>
                        <SelectItem value="client">Client (Portail B2B)</SelectItem>
                      </SelectContent>
                    </Select>
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="language">Langue</Label>
                    <Select
                      value={formData.language}
                      onValueChange={(value) => setFormData({...formData, language: value})}
                    >
                      <SelectTrigger>
                        <SelectValue />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="fr">Français</SelectItem>
                        <SelectItem value="ar">العربية</SelectItem>
                        <SelectItem value="en">English</SelectItem>
                      </SelectContent>
                    </Select>
                  </div>
                </div>
              </TabsContent>

              <TabsContent value="permissions" className="space-y-4 mt-4">
                <div className="bg-amber-50 border border-amber-200 rounded-lg p-4 mb-4">
                  <h4 className="font-semibold text-amber-900 mb-2">Templates Prédéfinis</h4>
                  <p className="text-sm text-amber-800 mb-3">
                    Sélectionnez un template pour appliquer rapidement un ensemble de permissions
                  </p>
                  <div className="grid grid-cols-3 gap-2">
                    {Object.entries(templates).map(([key, template]) => (
                      <Button
                        key={key}
                        type="button"
                        variant={selectedTemplate === key ? "default" : "outline"}
                        size="sm"
                        onClick={() => handleApplyTemplate(key)}
                        className="text-xs"
                      >
                        {template.name}
                      </Button>
                    ))}
                  </div>
                </div>

                <PermissionsSelector
                  permissions={permissionsData}
                  onChange={setPermissionsData}
                />
              </TabsContent>
            </Tabs>

            <DialogFooter className="mt-6">
              <Button 
                type="button" 
                variant="outline" 
                onClick={() => {
                  setShowDialog(false);
                  resetForm();
                }}
              >
                Annuler
              </Button>
              <Button 
                type="submit"
                className="bg-indigo-600 hover:bg-indigo-700"
              >
                {editMode ? 'Modifier' : 'Créer'}
              </Button>
            </DialogFooter>
          </form>
        </DialogContent>
      </Dialog>

      {/* Permissions Edit Dialog */}
      <Dialog open={showPermissionsDialog} onOpenChange={setShowPermissionsDialog}>
        <DialogContent className="max-w-5xl max-h-[90vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle>
              <div className="flex items-center gap-2">
                <Shield className="h-5 w-5" />
                Gérer les Permissions - {selectedUser?.full_name}
              </div>
            </DialogTitle>
            <DialogDescription>
              Définissez les permissions spécifiques pour cet utilisateur
            </DialogDescription>
          </DialogHeader>

          <div className="space-y-4">
            <div className="bg-amber-50 border border-amber-200 rounded-lg p-4">
              <h4 className="font-semibold text-amber-900 mb-2">Templates Prédéfinis</h4>
              <div className="grid grid-cols-3 gap-2">
                {Object.entries(templates).map(([key, template]) => (
                  <Button
                    key={key}
                    type="button"
                    variant={selectedTemplate === key ? "default" : "outline"}
                    size="sm"
                    onClick={() => handleApplyTemplate(key)}
                    className="text-xs"
                  >
                    {template.name}
                  </Button>
                ))}
              </div>
            </div>

            <PermissionsSelector
              permissions={permissionsData}
              onChange={setPermissionsData}
            />
          </div>

          <DialogFooter>
            <Button 
              variant="outline" 
              onClick={() => {
                setShowPermissionsDialog(false);
                setSelectedUser(null);
                setPermissionsData(null);
                setSelectedTemplate('');
              }}
            >
              Annuler
            </Button>
            <Button 
              onClick={handleSavePermissions}
              className="bg-indigo-600 hover:bg-indigo-700"
            >
              Enregistrer les Permissions
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
};

export default UsersPage;
