import React, { useEffect, useState } from 'react';
import api from '@/services/api';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Badge } from '@/components/ui/badge';
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle, DialogTrigger } from '@/components/ui/dialog';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Search, Eye, Package, TrendingUp } from 'lucide-react';
import { format } from 'date-fns';

const OrdersPage = () => {
  const [orders, setOrders] = useState([]);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState('');
  const [selectedOrder, setSelectedOrder] = useState(null);
  const [showDetails, setShowDetails] = useState(false);

  useEffect(() => {
    fetchOrders();
  }, []);

  const fetchOrders = async () => {
    try {
      const response = await api.get('/orders');
      setOrders(response.data);
    } catch (error) {
      console.error('Error fetching orders:', error);
    } finally {
      setLoading(false);
    }
  };

  const updateOrderStatus = async (orderId, newStatus) => {
    try {
      await api.patch(`/orders/${orderId}/status`, { status: newStatus });
      fetchOrders();
    } catch (error) {
      console.error('Error updating order status:', error);
    }
  };

  const getStatusBadge = (status) => {
    const statusConfig = {
      pending: { variant: 'warning', label: 'En attente' },
      confirmed: { variant: 'info', label: 'Confirmée' },
      preparing: { variant: 'info', label: 'En préparation' },
      ready: { variant: 'success', label: 'Prête' },
      in_delivery: { variant: 'info', label: 'En livraison' },
      delivered: { variant: 'success', label: 'Livrée' },
      cancelled: { variant: 'destructive', label: 'Annulée' },
    };
    
    const config = statusConfig[status] || statusConfig.pending;
    return <Badge variant={config.variant}>{config.label}</Badge>;
  };

  const filteredOrders = orders.filter(order =>
    order.order_number.toLowerCase().includes(searchTerm.toLowerCase()) ||
    order.client_name.toLowerCase().includes(searchTerm.toLowerCase())
  );

  const stats = {
    total: orders.length,
    pending: orders.filter(o => o.status === 'pending').length,
    revenue: orders.filter(o => o.status === 'delivered').reduce((sum, o) => sum + o.total, 0),
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center h-64">
        <div className="text-lg">Chargement...</div>
      </div>
    );
  }

  return (
    <div className="space-y-6" data-testid="orders-page">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold text-gray-900">Gestion des Commandes</h1>
          <p className="text-gray-600 mt-2">{orders.length} commande(s) au total</p>
        </div>
      </div>

      {/* Stats Cards */}
      <div className="grid md:grid-cols-3 gap-6">
        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Total Commandes</CardTitle>
            <Package className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{stats.total}</div>
            <p className="text-xs text-muted-foreground">Toutes les commandes</p>
          </CardContent>
        </Card>
        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">En Attente</CardTitle>
            <Package className="h-4 w-4 text-yellow-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-yellow-600">{stats.pending}</div>
            <p className="text-xs text-muted-foreground">À traiter</p>
          </CardContent>
        </Card>
        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">CA Total</CardTitle>
            <TrendingUp className="h-4 w-4 text-green-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-green-600">
              {new Intl.NumberFormat('fr-DZ', {
                style: 'currency',
                currency: 'DZD',
                minimumFractionDigits: 0,
              }).format(stats.revenue)}
            </div>
            <p className="text-xs text-muted-foreground">Commandes livrées</p>
          </CardContent>
        </Card>
      </div>

      <Card>
        <CardHeader>
          <div className="flex items-center justify-between">
            <div>
              <CardTitle>Liste des Commandes</CardTitle>
              <CardDescription>Gérez toutes les commandes clients</CardDescription>
            </div>
            <div className="relative">
              <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-gray-400" />
              <Input
                placeholder="Rechercher..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="pl-10 w-64"
                data-testid="search-orders-input"
              />
            </div>
          </div>
        </CardHeader>
        <CardContent>
          <div className="overflow-x-auto">
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead>N° Commande</TableHead>
                  <TableHead>Client</TableHead>
                  <TableHead>Date</TableHead>
                  <TableHead>Articles</TableHead>
                  <TableHead className="text-right">Total</TableHead>
                  <TableHead>Statut</TableHead>
                  <TableHead className="text-center">Actions</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {filteredOrders.length === 0 ? (
                  <TableRow>
                    <TableCell colSpan={7} className="text-center py-8 text-gray-500">
                      Aucune commande trouvée
                    </TableCell>
                  </TableRow>
                ) : (
                  filteredOrders.map((order) => (
                    <TableRow key={order.id} data-testid={`order-row-${order.id}`}>
                      <TableCell className="font-medium">{order.order_number}</TableCell>
                      <TableCell>{order.client_name}</TableCell>
                      <TableCell>
                        {order.created_at ? format(new Date(order.created_at), 'dd/MM/yyyy') : '-'}
                      </TableCell>
                      <TableCell>{order.items?.length || 0} article(s)</TableCell>
                      <TableCell className="text-right font-semibold">
                        {new Intl.NumberFormat('fr-DZ', {
                          style: 'currency',
                          currency: 'DZD',
                        }).format(order.total)}
                      </TableCell>
                      <TableCell>{getStatusBadge(order.status)}</TableCell>
                      <TableCell>
                        <div className="flex items-center justify-center space-x-2">
                          <Dialog open={showDetails && selectedOrder?.id === order.id} onOpenChange={(open) => {
                            setShowDetails(open);
                            if (!open) setSelectedOrder(null);
                          }}>
                            <DialogTrigger asChild>
                              <Button
                                variant="ghost"
                                size="sm"
                                onClick={() => setSelectedOrder(order)}
                                data-testid={`view-order-${order.id}`}
                              >
                                <Eye className="h-4 w-4" />
                              </Button>
                            </DialogTrigger>
                            <DialogContent className="max-w-2xl">
                              <DialogHeader>
                                <DialogTitle>Détails de la commande {order.order_number}</DialogTitle>
                                <DialogDescription>
                                  Client: {order.client_name}
                                </DialogDescription>
                              </DialogHeader>
                              <div className="space-y-4">
                                <div className="grid grid-cols-2 gap-4">
                                  <div>
                                    <p className="text-sm text-gray-500">Statut</p>
                                    <div className="mt-1">{getStatusBadge(order.status)}</div>
                                  </div>
                                  <div>
                                    <p className="text-sm text-gray-500">Date</p>
                                    <p className="font-medium">
                                      {order.created_at ? format(new Date(order.created_at), 'dd/MM/yyyy HH:mm') : '-'}
                                    </p>
                                  </div>
                                </div>

                                <div>
                                  <p className="text-sm text-gray-500 mb-2">Articles commandés</p>
                                  <div className="border rounded-lg overflow-hidden">
                                    <table className="w-full text-sm">
                                      <thead className="bg-gray-50">
                                        <tr>
                                          <th className="px-4 py-2 text-left">Produit</th>
                                          <th className="px-4 py-2 text-center">Qté</th>
                                          <th className="px-4 py-2 text-right">Prix Unit.</th>
                                          <th className="px-4 py-2 text-right">Total</th>
                                        </tr>
                                      </thead>
                                      <tbody>
                                        {order.items?.map((item, idx) => (
                                          <tr key={idx} className="border-t">
                                            <td className="px-4 py-2">
                                              <p className="font-medium">{item.product_name}</p>
                                              <p className="text-xs text-gray-500">{item.product_reference}</p>
                                            </td>
                                            <td className="px-4 py-2 text-center">{item.quantity}</td>
                                            <td className="px-4 py-2 text-right">
                                              {new Intl.NumberFormat('fr-DZ', {
                                                style: 'currency',
                                                currency: 'DZD',
                                              }).format(item.unit_price)}
                                            </td>
                                            <td className="px-4 py-2 text-right font-semibold">
                                              {new Intl.NumberFormat('fr-DZ', {
                                                style: 'currency',
                                                currency: 'DZD',
                                              }).format(item.subtotal)}
                                            </td>
                                          </tr>
                                        ))}
                                      </tbody>
                                    </table>
                                  </div>
                                </div>

                                <div className="border-t pt-4">
                                  <div className="space-y-2">
                                    <div className="flex justify-between">
                                      <span>Sous-total</span>
                                      <span className="font-medium">
                                        {new Intl.NumberFormat('fr-DZ', {
                                          style: 'currency',
                                          currency: 'DZD',
                                        }).format(order.subtotal)}
                                      </span>
                                    </div>
                                    <div className="flex justify-between">
                                      <span>TVA (19%)</span>
                                      <span className="font-medium">
                                        {new Intl.NumberFormat('fr-DZ', {
                                          style: 'currency',
                                          currency: 'DZD',
                                        }).format(order.tax_amount)}
                                      </span>
                                    </div>
                                    <div className="flex justify-between text-lg font-bold border-t pt-2">
                                      <span>Total</span>
                                      <span className="text-indigo-600">
                                        {new Intl.NumberFormat('fr-DZ', {
                                          style: 'currency',
                                          currency: 'DZD',
                                        }).format(order.total)}
                                      </span>
                                    </div>
                                  </div>
                                </div>

                                {order.status !== 'delivered' && order.status !== 'cancelled' && (
                                  <div>
                                    <p className="text-sm text-gray-500 mb-2">Changer le statut</p>
                                    <Select onValueChange={(value) => {
                                      updateOrderStatus(order.id, value);
                                      setShowDetails(false);
                                    }}>
                                      <SelectTrigger>
                                        <SelectValue placeholder="Sélectionner un statut" />
                                      </SelectTrigger>
                                      <SelectContent>
                                        <SelectItem value="confirmed">Confirmée</SelectItem>
                                        <SelectItem value="preparing">En préparation</SelectItem>
                                        <SelectItem value="ready">Prête</SelectItem>
                                        <SelectItem value="in_delivery">En livraison</SelectItem>
                                        <SelectItem value="delivered">Livrée</SelectItem>
                                        <SelectItem value="cancelled">Annuler</SelectItem>
                                      </SelectContent>
                                    </Select>
                                  </div>
                                )}
                              </div>
                              <DialogFooter>
                                <Button variant="outline" onClick={() => setShowDetails(false)}>
                                  Fermer
                                </Button>
                              </DialogFooter>
                            </DialogContent>
                          </Dialog>
                        </div>
                      </TableCell>
                    </TableRow>
                  ))
                )}
              </TableBody>
            </Table>
          </div>
        </CardContent>
      </Card>
    </div>
  );
};

export default OrdersPage;
