import React, { useState } from 'react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Badge } from '@/components/ui/badge';
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle, DialogTrigger } from '@/components/ui/dialog';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Ship, Plus, FileText, DollarSign, Calculator } from 'lucide-react';

const ImportsManagement = () => {
  // Les conteneurs avec leurs statuts de workflow
  // Statuts possibles: ordered → en_transit → arrived → customs → cleared → delivered
  const [containers, setContainers] = useState([
    {
      id: 'CONT-001',
      reference: 'MSCU1234567',
      supplier: 'Auto Parts International - Chine',
      departure_port: 'Shanghai',
      arrival_port: 'Alger',
      departure_date: '2025-01-15',
      arrival_date: '2025-02-20',
      status: 'en_transit', // Le conteneur est actuellement en mer
      fob_value: 45000,
      freight_cost: 8500,
      insurance: 1200,
      customs_duties: 12500,
      other_costs: 2300,
    },
    {
      id: 'CONT-002',
      reference: 'HLCU9876543',
      supplier: 'European Auto Parts - Allemagne',
      departure_port: 'Hamburg',
      arrival_port: 'Oran',
      departure_date: '2025-01-20',
      arrival_date: '2025-02-10',
      status: 'customs', // Le conteneur est en cours de dédouanement
      fob_value: 65000,
      freight_cost: 6200,
      insurance: 1800,
      customs_duties: 18200,
      other_costs: 3100,
    },
  ]);

  const [showDialog, setShowDialog] = useState(false);
  const [showCalculator, setShowCalculator] = useState(false);
  const [showStatusDialog, setShowStatusDialog] = useState(false);
  const [selectedContainer, setSelectedContainer] = useState(null);
  const [calcData, setCalcData] = useState({
    fob: 0,
    freight: 0,
    insurance: 0,
    customs_rate: 30,
    vat_rate: 19,
  });

  const [formData, setFormData] = useState({
    reference: '',
    supplier: '',
    departure_port: '',
    arrival_port: 'Alger',
    departure_date: '',
    fob_value: 0,
  });

  const getStatusBadge = (status) => {
    const config = {
      ordered: { variant: 'outline', label: 'Commandé' },
      en_transit: { variant: 'info', label: 'En Transit' },
      arrived: { variant: 'warning', label: 'Arrivé' },
      customs: { variant: 'warning', label: 'En Douane' },
      cleared: { variant: 'success', label: 'Dédouané' },
      delivered: { variant: 'success', label: 'Livré' },
    };
    const c = config[status] || config.ordered;
    return <Badge variant={c.variant}>{c.label}</Badge>;
  };

  const calculateCIF = (container) => {
    return container.fob_value + container.freight_cost + container.insurance;
  };

  const calculateTotalCost = (container) => {
    const cif = calculateCIF(container);
    return cif + container.customs_duties + container.other_costs;
  };

  const calculateImportCosts = () => {
    const fob = parseFloat(calcData.fob) || 0;
    const freight = parseFloat(calcData.freight) || 0;
    const insurance = parseFloat(calcData.insurance) || 0;
    
    const cif = fob + freight + insurance;
    const customs = cif * (calcData.customs_rate / 100);
    const taxBase = cif + customs;
    const vat = taxBase * (calcData.vat_rate / 100);
    const total = taxBase + vat;

    return { cif, customs, taxBase, vat, total };
  };

  const costs = calculateImportCosts();

  const totalInTransit = containers
    .filter(c => c.status === 'en_transit' || c.status === 'ordered')
    .reduce((sum, c) => sum + calculateTotalCost(c), 0);

  const updateContainerStatus = (containerId, newStatus) => {
    setContainers(containers.map(c => 
      c.id === containerId ? {...c, status: newStatus} : c
    ));
    setShowStatusDialog(false);
  };

  const openStatusDialog = (container) => {
    setSelectedContainer(container);
    setShowStatusDialog(true);
  };

  return (
    <div className="space-y-6" data-testid="imports-page">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold text-gray-900">Gestion Imports & Conteneurs</h1>
          <p className="text-gray-600 mt-2">Suivez vos importations internationales</p>
        </div>
        <div className="flex gap-2">
          <Button
            variant="outline"
            onClick={() => setShowCalculator(true)}
            data-testid="calculator-button"
          >
            <Calculator className="h-4 w-4 mr-2" />
            Calculateur
          </Button>
          <Button
            className="bg-indigo-600 hover:bg-indigo-700"
            onClick={() => setShowDialog(true)}
            data-testid="add-container-button"
          >
            <Plus className="h-4 w-4 mr-2" />
            Nouveau Conteneur
          </Button>
        </div>
      </div>

      {/* Stats Cards */}
      <div className="grid gap-6 md:grid-cols-4">
        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Conteneurs Actifs</CardTitle>
            <Ship className="h-4 w-4 text-blue-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{containers.length}</div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">En Transit</CardTitle>
            <Ship className="h-4 w-4 text-orange-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-orange-600">
              {containers.filter(c => c.status === 'en_transit').length}
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Valeur en Transit</CardTitle>
            <DollarSign className="h-4 w-4 text-green-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-green-600">
              {new Intl.NumberFormat('fr-DZ', {
                style: 'currency',
                currency: 'DZD',
                minimumFractionDigits: 0,
              }).format(totalInTransit)}
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">En Douane</CardTitle>
            <FileText className="h-4 w-4 text-yellow-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-yellow-600">
              {containers.filter(c => c.status === 'customs').length}
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Containers Table */}
      <Card>
        <CardHeader>
          <CardTitle>Liste des Conteneurs</CardTitle>
          <CardDescription>Suivi de vos importations</CardDescription>
        </CardHeader>
        <CardContent>
          <div className="overflow-x-auto">
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead>Référence</TableHead>
                  <TableHead>Fournisseur</TableHead>
                  <TableHead>Trajet</TableHead>
                  <TableHead>Dates</TableHead>
                  <TableHead className="text-right">FOB (USD)</TableHead>
                  <TableHead className="text-right">CIF (USD)</TableHead>
                  <TableHead className="text-right">Coût Total</TableHead>
                  <TableHead>Statut</TableHead>
                  <TableHead>Actions</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {containers.map((container) => (
                  <TableRow key={container.id}>
                    <TableCell className="font-medium">{container.reference}</TableCell>
                    <TableCell>{container.supplier}</TableCell>
                    <TableCell>
                      <div className="text-sm">
                        <p>{container.departure_port} → {container.arrival_port}</p>
                      </div>
                    </TableCell>
                    <TableCell>
                      <div className="text-sm">
                        <p>Départ: {new Date(container.departure_date).toLocaleDateString('fr-FR')}</p>
                        <p className="text-gray-500">Arrivée: {new Date(container.arrival_date).toLocaleDateString('fr-FR')}</p>
                      </div>
                    </TableCell>
                    <TableCell className="text-right font-semibold">
                      ${container.fob_value.toLocaleString()}
                    </TableCell>
                    <TableCell className="text-right font-semibold text-blue-600">
                      ${calculateCIF(container).toLocaleString()}
                    </TableCell>
                    <TableCell className="text-right font-bold text-green-600">
                      {new Intl.NumberFormat('fr-DZ', {
                        style: 'currency',
                        currency: 'DZD',
                        minimumFractionDigits: 0,
                      }).format(calculateTotalCost(container))}
                    </TableCell>
                    <TableCell>{getStatusBadge(container.status)}</TableCell>
                    <TableCell>
                      <Button
                        variant="outline"
                        size="sm"
                        onClick={() => openStatusDialog(container)}
                      >
                        Changer statut
                      </Button>
                    </TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          </div>
        </CardContent>
      </Card>

      {/* Cost Calculator Dialog */}
      <Dialog open={showCalculator} onOpenChange={setShowCalculator}>
        <DialogContent className="max-w-2xl">
          <DialogHeader>
            <DialogTitle>Calculateur de Coûts d'Import</DialogTitle>
            <DialogDescription>
              Calculez le coût total d'une importation (FOB → CIF → Dédouané)
            </DialogDescription>
          </DialogHeader>
          <div className="grid gap-4 py-4">
            <div className="grid grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label>FOB (USD)</Label>
                <Input
                  type="number"
                  value={calcData.fob}
                  onChange={(e) => setCalcData({...calcData, fob: e.target.value})}
                  placeholder="45000"
                />
              </div>
              <div className="space-y-2">
                <Label>Fret (USD)</Label>
                <Input
                  type="number"
                  value={calcData.freight}
                  onChange={(e) => setCalcData({...calcData, freight: e.target.value})}
                  placeholder="8500"
                />
              </div>
            </div>

            <div className="grid grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label>Assurance (USD)</Label>
                <Input
                  type="number"
                  value={calcData.insurance}
                  onChange={(e) => setCalcData({...calcData, insurance: e.target.value})}
                  placeholder="1200"
                />
              </div>
              <div className="space-y-2">
                <Label>Taux Douane (%)</Label>
                <Input
                  type="number"
                  value={calcData.customs_rate}
                  onChange={(e) => setCalcData({...calcData, customs_rate: e.target.value})}
                />
              </div>
            </div>

            <div className="space-y-2">
              <Label>Taux TVA (%)</Label>
              <Input
                type="number"
                value={calcData.vat_rate}
                onChange={(e) => setCalcData({...calcData, vat_rate: e.target.value})}
              />
            </div>

            <div className="border-t pt-4 space-y-3">
              <div className="flex justify-between text-base">
                <span>CIF (FOB + Fret + Assurance):</span>
                <span className="font-bold">${costs.cif.toLocaleString()}</span>
              </div>
              <div className="flex justify-between text-base">
                <span>Droits de Douane ({calcData.customs_rate}%):</span>
                <span className="font-bold">${costs.customs.toLocaleString()}</span>
              </div>
              <div className="flex justify-between text-base">
                <span>Base Taxable:</span>
                <span className="font-bold">${costs.taxBase.toLocaleString()}</span>
              </div>
              <div className="flex justify-between text-base">
                <span>TVA ({calcData.vat_rate}%):</span>
                <span className="font-bold">${costs.vat.toLocaleString()}</span>
              </div>
              <div className="flex justify-between text-xl font-bold border-t pt-3 text-green-600">
                <span>COÛT TOTAL DÉDOUANÉ:</span>
                <span>${costs.total.toLocaleString()}</span>
              </div>
            </div>
          </div>
          <DialogFooter>
            <Button variant="outline" onClick={() => setShowCalculator(false)}>Fermer</Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Add Container Dialog */}
      <Dialog open={showDialog} onOpenChange={setShowDialog}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Nouveau Conteneur</DialogTitle>
            <DialogDescription>Enregistrer un nouveau conteneur d'import</DialogDescription>
          </DialogHeader>
          <div className="grid gap-4 py-4">
            <div className="space-y-2">
              <Label>Référence Conteneur</Label>
              <Input
                value={formData.reference}
                onChange={(e) => setFormData({...formData, reference: e.target.value})}
                placeholder="MSCU1234567"
              />
            </div>
            <div className="space-y-2">
              <Label>Fournisseur</Label>
              <Input
                value={formData.supplier}
                onChange={(e) => setFormData({...formData, supplier: e.target.value})}
                placeholder="Nom du fournisseur"
              />
            </div>
            <div className="grid grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label>Port Départ</Label>
                <Input
                  value={formData.departure_port}
                  onChange={(e) => setFormData({...formData, departure_port: e.target.value})}
                  placeholder="Shanghai"
                />
              </div>
              <div className="space-y-2">
                <Label>Port Arrivée</Label>
                <Select
                  value={formData.arrival_port}
                  onValueChange={(value) => setFormData({...formData, arrival_port: value})}
                >
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="Alger">Alger</SelectItem>
                    <SelectItem value="Oran">Oran</SelectItem>
                    <SelectItem value="Annaba">Annaba</SelectItem>
                  </SelectContent>
                </Select>
              </div>
            </div>
            <div className="space-y-2">
              <Label>Valeur FOB (USD)</Label>
              <Input
                type="number"
                value={formData.fob_value}
                onChange={(e) => setFormData({...formData, fob_value: parseFloat(e.target.value)})}
                placeholder="45000"
              />
            </div>
          </div>
          <DialogFooter>
            <Button variant="outline" onClick={() => setShowDialog(false)}>Annuler</Button>
            <Button className="bg-indigo-600">Créer</Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Status Update Dialog */}
      <Dialog open={showStatusDialog} onOpenChange={setShowStatusDialog}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Changer le Statut du Conteneur</DialogTitle>
            <DialogDescription>
              {selectedContainer?.reference} - {selectedContainer?.supplier}
            </DialogDescription>
          </DialogHeader>
          <div className="py-4">
            <div className="space-y-2">
              <Label>Nouveau Statut</Label>
              <div className="space-y-2">
                <p className="text-sm text-gray-500 mb-3">
                  Statut actuel: {selectedContainer && getStatusBadge(selectedContainer.status)}
                </p>
                <div className="grid gap-2">
                  <Button
                    variant="outline"
                    className="justify-start"
                    onClick={() => updateContainerStatus(selectedContainer?.id, 'ordered')}
                  >
                    <Badge variant="outline" className="mr-2">Commandé</Badge>
                    Le conteneur a été commandé
                  </Button>
                  <Button
                    variant="outline"
                    className="justify-start"
                    onClick={() => updateContainerStatus(selectedContainer?.id, 'en_transit')}
                  >
                    <Badge variant="default" className="mr-2">En Transit</Badge>
                    Le conteneur est en mer
                  </Button>
                  <Button
                    variant="outline"
                    className="justify-start"
                    onClick={() => updateContainerStatus(selectedContainer?.id, 'arrived')}
                  >
                    <Badge variant="secondary" className="mr-2">Arrivé</Badge>
                    Le conteneur est arrivé au port
                  </Button>
                  <Button
                    variant="outline"
                    className="justify-start"
                    onClick={() => updateContainerStatus(selectedContainer?.id, 'customs')}
                  >
                    <Badge variant="secondary" className="mr-2">En Douane</Badge>
                    Le conteneur est en cours de dédouanement
                  </Button>
                  <Button
                    variant="outline"
                    className="justify-start"
                    onClick={() => updateContainerStatus(selectedContainer?.id, 'cleared')}
                  >
                    <Badge variant="default" className="mr-2">Dédouané</Badge>
                    Le conteneur a été dédouané
                  </Button>
                  <Button
                    variant="outline"
                    className="justify-start"
                    onClick={() => updateContainerStatus(selectedContainer?.id, 'delivered')}
                  >
                    <Badge variant="default" className="mr-2">Livré</Badge>
                    Le conteneur a été livré au dépôt
                  </Button>
                </div>
              </div>
            </div>
          </div>
          <DialogFooter>
            <Button variant="outline" onClick={() => setShowStatusDialog(false)}>Fermer</Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
};

export default ImportsManagement;
