import React, { useState, useEffect } from 'react';
import api from '@/services/api';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Badge } from '@/components/ui/badge';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle } from '@/components/ui/dialog';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Textarea } from '@/components/ui/textarea';
import { Users, Calendar, DollarSign, GraduationCap, Plus, CheckCircle, XCircle, Eye, Edit, Trash2, UserX, Search, FileText, Clock } from 'lucide-react';

const HRPage = () => {
  const [dashboard, setDashboard] = useState({
    total_employees: 0,
    departments_count: 0,
    pending_leave_requests: 0,
    today_attendance: 0,
    upcoming_trainings: 0
  });
  const [employees, setEmployees] = useState([]);
  const [leaveRequests, setLeaveRequests] = useState([]);
  const [attendances, setAttendances] = useState([]);
  const [payslips, setPayslips] = useState([]);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState('');
  const [filterStatus, setFilterStatus] = useState('');
  const [filterDepartment, setFilterDepartment] = useState('');
  
  // Dialogs
  const [showEmployeeDialog, setShowEmployeeDialog] = useState(false);
  const [showEmployeeDetailsDialog, setShowEmployeeDetailsDialog] = useState(false);
  const [showEditEmployeeDialog, setShowEditEmployeeDialog] = useState(false);
  const [showTerminateDialog, setShowTerminateDialog] = useState(false);
  const [showLeaveRequestDialog, setShowLeaveRequestDialog] = useState(false);
  const [showAttendanceDialog, setShowAttendanceDialog] = useState(false);
  const [showGeneratePayslipDialog, setShowGeneratePayslipDialog] = useState(false);
  const [showPayslipDetailsDialog, setShowPayslipDetailsDialog] = useState(false);
  
  // Selected items
  const [selectedEmployee, setSelectedEmployee] = useState(null);
  const [selectedPayslip, setSelectedPayslip] = useState(null);
  
  // Forms
  const [employeeForm, setEmployeeForm] = useState({
    first_name: '',
    last_name: '',
    date_of_birth: '',
    place_of_birth: '',
    gender: 'M',
    cin_number: '',
    social_security_number: '',
    phone: '',
    email: '',
    address: '',
    city: '',
    employee_number: '',
    hire_date: '',
    contract_type: 'cdi',
    position: '',
    department: '',
    base_salary: 0
  });

  const [terminateForm, setTerminateForm] = useState({
    termination_date: new Date().toISOString().split('T')[0],
    termination_reason: ''
  });

  const [leaveRequestForm, setLeaveRequestForm] = useState({
    employee_id: '',
    leave_type: 'annual',
    start_date: '',
    end_date: '',
    reason: ''
  });

  const [attendanceForm, setAttendanceForm] = useState({
    employee_id: '',
    date: new Date().toISOString().split('T')[0],
    check_in: '08:00',
    check_out: '',
    notes: ''
  });

  const [generatePayslipForm, setGeneratePayslipForm] = useState({
    employee_id: '',
    period_month: new Date().getMonth() + 1,
    period_year: new Date().getFullYear()
  });

  useEffect(() => {
    fetchDashboard();
    fetchEmployees();
    fetchLeaveRequests();
  }, []);

  const fetchDashboard = async () => {
    try {
      const response = await api.get('/hr/dashboard');
      setDashboard(response.data);
    } catch (error) {
      console.error('Error:', error);
    }
  };

  const fetchEmployees = async () => {
    try {
      const response = await api.get('/hr/employees?status=active');
      setEmployees(response.data);
    } catch (error) {
      console.error('Error:', error);
    } finally {
      setLoading(false);
    }
  };

  const fetchLeaveRequests = async () => {
    try {
      const response = await api.get('/hr/leave-requests?status=pending');
      setLeaveRequests(response.data);
    } catch (error) {
      console.error('Error:', error);
    }
  };

  const fetchAttendances = async () => {
    try {
      const today = new Date().toISOString().split('T')[0];
      const response = await api.get(`/hr/attendance?date=${today}`);
      setAttendances(response.data);
    } catch (error) {
      console.error('Error:', error);
    }
  };

  const fetchPayslips = async () => {
    try {
      const response = await api.get('/hr/payslips');
      setPayslips(response.data);
    } catch (error) {
      console.error('Error:', error);
    }
  };

  // Employee handlers
  const handleCreateEmployee = async () => {
    try {
      await api.post('/hr/employees', employeeForm);
      setShowEmployeeDialog(false);
      resetEmployeeForm();
      fetchEmployees();
      fetchDashboard();
      alert('Employé créé avec succès');
    } catch (error) {
      console.error('Error:', error);
      alert(error.response?.data?.detail || 'Erreur lors de la création');
    }
  };

  const handleUpdateEmployee = async () => {
    try {
      await api.put(`/hr/employees/${selectedEmployee.id}`, employeeForm);
      setShowEditEmployeeDialog(false);
      resetEmployeeForm();
      setSelectedEmployee(null);
      fetchEmployees();
      alert('Employé modifié avec succès');
    } catch (error) {
      console.error('Error:', error);
      alert('Erreur lors de la modification');
    }
  };

  const handleTerminateEmployee = async () => {
    if (!window.confirm(`Êtes-vous sûr de vouloir mettre fin au contrat de ${selectedEmployee.first_name} ${selectedEmployee.last_name} ?`)) {
      return;
    }
    try {
      await api.patch(`/hr/employees/${selectedEmployee.id}/terminate`, terminateForm);
      setShowTerminateDialog(false);
      setSelectedEmployee(null);
      resetTerminateForm();
      fetchEmployees();
      fetchDashboard();
      alert('Contrat résilié avec succès');
    } catch (error) {
      console.error('Error:', error);
      alert('Erreur lors de la résiliation');
    }
  };

  const handleViewEmployee = (employee) => {
    setSelectedEmployee(employee);
    setShowEmployeeDetailsDialog(true);
  };

  const handleEditEmployee = (employee) => {
    setSelectedEmployee(employee);
    setEmployeeForm({
      first_name: employee.first_name || '',
      last_name: employee.last_name || '',
      date_of_birth: employee.date_of_birth ? new Date(employee.date_of_birth).toISOString().split('T')[0] : '',
      place_of_birth: employee.place_of_birth || '',
      gender: employee.gender || 'M',
      cin_number: employee.cin_number || '',
      social_security_number: employee.social_security_number || '',
      phone: employee.phone || '',
      email: employee.email || '',
      address: employee.address || '',
      city: employee.city || '',
      employee_number: employee.employee_number || '',
      hire_date: employee.hire_date ? new Date(employee.hire_date).toISOString().split('T')[0] : '',
      contract_type: employee.contract_type || 'cdi',
      position: employee.position || '',
      department: employee.department || '',
      base_salary: employee.base_salary || 0
    });
    setShowEditEmployeeDialog(true);
  };

  const handleOpenTerminate = (employee) => {
    setSelectedEmployee(employee);
    setShowTerminateDialog(true);
  };

  // Leave request handlers
  const handleCreateLeaveRequest = async () => {
    try {
      await api.post('/hr/leave-requests', leaveRequestForm);
      setShowLeaveRequestDialog(false);
      resetLeaveRequestForm();
      fetchLeaveRequests();
      fetchDashboard();
      alert('Demande de congé créée avec succès');
    } catch (error) {
      console.error('Error:', error);
      alert(error.response?.data?.detail || 'Erreur lors de la création');
    }
  };

  const handleApproveLeave = async (requestId) => {
    if (!window.confirm('Approuver cette demande de congé ?')) return;
    
    try {
      await api.patch(`/hr/leave-requests/${requestId}/approve`);
      fetchLeaveRequests();
      fetchDashboard();
      alert('Demande approuvée');
    } catch (error) {
      console.error('Error:', error);
      alert('Erreur lors de l\'approbation');
    }
  };

  const handleRejectLeave = async (requestId) => {
    const reason = prompt('Raison du rejet:');
    if (!reason) return;
    
    try {
      await api.patch(`/hr/leave-requests/${requestId}/reject?rejection_reason=${encodeURIComponent(reason)}`);
      fetchLeaveRequests();
      fetchDashboard();
      alert('Demande rejetée');
    } catch (error) {
      console.error('Error:', error);
      alert('Erreur lors du rejet');
    }
  };

  // Payslip handlers
  const handleGeneratePayslip = async () => {
    try {
      const response = await api.post('/hr/payslips/generate', null, {
        params: {
          employee_id: generatePayslipForm.employee_id,
          period_month: generatePayslipForm.period_month,
          period_year: generatePayslipForm.period_year
        }
      });
      setShowGeneratePayslipDialog(false);
      resetGeneratePayslipForm();
      fetchPayslips();
      alert('Bulletin de paie généré avec succès');
    } catch (error) {
      console.error('Error:', error);
      alert(error.response?.data?.detail || 'Erreur lors de la génération');
    }
  };

  const handleViewPayslip = (payslip) => {
    setSelectedPayslip(payslip);
    setShowPayslipDetailsDialog(true);
  };

  // Reset forms
  const resetEmployeeForm = () => {
    setEmployeeForm({
      first_name: '',
      last_name: '',
      date_of_birth: '',
      place_of_birth: '',
      gender: 'M',
      cin_number: '',
      social_security_number: '',
      phone: '',
      email: '',
      address: '',
      city: '',
      employee_number: '',
      hire_date: '',
      contract_type: 'cdi',
      position: '',
      department: '',
      base_salary: 0
    });
  };

  const resetTerminateForm = () => {
    setTerminateForm({
      termination_date: new Date().toISOString().split('T')[0],
      termination_reason: ''
    });
  };

  const resetLeaveRequestForm = () => {
    setLeaveRequestForm({
      employee_id: '',
      leave_type: 'annual',
      start_date: '',
      end_date: '',
      reason: ''
    });
  };

  const resetGeneratePayslipForm = () => {
    setGeneratePayslipForm({
      employee_id: '',
      period_month: new Date().getMonth() + 1,
      period_year: new Date().getFullYear()
    });
  };

  // Utility functions
  const getStatusBadge = (status) => {
    const badges = {
      active: { variant: 'success', label: 'Actif', class: 'bg-green-100 text-green-800' },
      on_leave: { variant: 'secondary', label: 'En congé', class: 'bg-yellow-100 text-yellow-800' },
      suspended: { variant: 'destructive', label: 'Suspendu', class: 'bg-red-100 text-red-800' },
      terminated: { variant: 'outline', label: 'Parti', class: 'bg-gray-100 text-gray-800' }
    };
    const b = badges[status] || badges.active;
    return <Badge className={b.class}>{b.label}</Badge>;
  };

  const getLeaveTypeBadge = (type) => {
    const types = {
      annual: 'Congé annuel',
      sick: 'Maladie',
      maternity: 'Maternité',
      paternity: 'Paternité',
      unpaid: 'Sans solde',
      exceptional: 'Exceptionnel'
    };
    return types[type] || type;
  };

  const getContractTypeLabel = (type) => {
    const types = {
      cdi: 'CDI',
      cdd: 'CDD',
      daip: 'DAIP',
      internship: 'Stage'
    };
    return types[type] || type;
  };

  const getMonthName = (month) => {
    const months = ['Janvier', 'Février', 'Mars', 'Avril', 'Mai', 'Juin', 
                    'Juillet', 'Août', 'Septembre', 'Octobre', 'Novembre', 'Décembre'];
    return months[month - 1] || month;
  };

  // Filter employees
  const filteredEmployees = employees.filter(emp => {
    const matchesSearch = searchTerm === '' || 
      `${emp.first_name} ${emp.last_name} ${emp.employee_number} ${emp.position}`.toLowerCase().includes(searchTerm.toLowerCase());
    const matchesStatus = !filterStatus || emp.status === filterStatus;
    const matchesDepartment = filterDepartment === '' || emp.department === filterDepartment;
    return matchesSearch && matchesStatus && matchesDepartment;
  });

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold text-gray-900">Ressources Humaines</h1>
          <p className="text-gray-600 mt-2">Gestion du personnel et paie</p>
        </div>
      </div>

      {/* Stats */}
      <div className="grid gap-6 md:grid-cols-4">
        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Effectif Total</CardTitle>
            <Users className="h-4 w-4 text-blue-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{dashboard.total_employees}</div>
            <p className="text-xs text-gray-500">{dashboard.departments_count} services</p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Présents Aujourd'hui</CardTitle>
            <Calendar className="h-4 w-4 text-green-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-green-600">{dashboard.today_attendance}</div>
            <p className="text-xs text-gray-500">Pointés</p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Congés en Attente</CardTitle>
            <Calendar className="h-4 w-4 text-orange-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-orange-600">{dashboard.pending_leave_requests}</div>
            <p className="text-xs text-gray-500">À valider</p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Formations à Venir</CardTitle>
            <GraduationCap className="h-4 w-4 text-purple-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{dashboard.upcoming_trainings}</div>
            <p className="text-xs text-gray-500">Planifiées</p>
          </CardContent>
        </Card>
      </div>

      <Tabs defaultValue="employees">
        <TabsList>
          <TabsTrigger value="employees">Employés</TabsTrigger>
          <TabsTrigger value="attendance">Pointage</TabsTrigger>
          <TabsTrigger value="leaves">Congés</TabsTrigger>
          <TabsTrigger value="payroll">Paie</TabsTrigger>
        </TabsList>

        {/* EMPLOYEES TAB */}
        <TabsContent value="employees">
          <Card>
            <CardHeader>
              <div className="flex items-center justify-between">
                <div>
                  <CardTitle>Liste des Employés</CardTitle>
                </div>
                <Button 
                  className="bg-indigo-600 hover:bg-indigo-700"
                  onClick={() => setShowEmployeeDialog(true)}
                >
                  <Plus className="h-4 w-4 mr-2" />
                  Nouvel Employé
                </Button>
              </div>
            </CardHeader>
            <CardContent>
              {/* Filters */}
              <div className="grid grid-cols-3 gap-4 mb-4">
                <div className="relative">
                  <Search className="absolute left-3 top-3 h-4 w-4 text-gray-400" />
                  <Input
                    placeholder="Rechercher..."
                    value={searchTerm}
                    onChange={(e) => setSearchTerm(e.target.value)}
                    className="pl-9"
                  />
                </div>
                <Select value={filterStatus || undefined} onValueChange={setFilterStatus}>
                  <SelectTrigger>
                    <SelectValue placeholder="Tous les statuts" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="active">Actif</SelectItem>
                    <SelectItem value="on_leave">En congé</SelectItem>
                    <SelectItem value="suspended">Suspendu</SelectItem>
                    <SelectItem value="terminated">Parti</SelectItem>
                  </SelectContent>
                </Select>
                <Input
                  placeholder="Filtrer par service..."
                  value={filterDepartment}
                  onChange={(e) => setFilterDepartment(e.target.value)}
                />
              </div>

              {loading ? (
                <div className="text-center py-8">Chargement...</div>
              ) : filteredEmployees.length === 0 ? (
                <div className="text-center py-8 text-gray-500">Aucun employé trouvé</div>
              ) : (
                <div className="overflow-x-auto">
                  <Table>
                    <TableHeader>
                      <TableRow>
                        <TableHead>Matricule</TableHead>
                        <TableHead>Nom Prénom</TableHead>
                        <TableHead>Poste</TableHead>
                        <TableHead>Service</TableHead>
                        <TableHead>Téléphone</TableHead>
                        <TableHead>Date d'embauche</TableHead>
                        <TableHead>Statut</TableHead>
                        <TableHead className="text-right">Actions</TableHead>
                      </TableRow>
                    </TableHeader>
                    <TableBody>
                      {filteredEmployees.map((emp) => (
                        <TableRow key={emp.id}>
                          <TableCell className="font-medium">{emp.employee_number}</TableCell>
                          <TableCell>{emp.first_name} {emp.last_name}</TableCell>
                          <TableCell>{emp.position}</TableCell>
                          <TableCell>{emp.department}</TableCell>
                          <TableCell>{emp.phone}</TableCell>
                          <TableCell>
                            {new Date(emp.hire_date).toLocaleDateString('fr-FR')}
                          </TableCell>
                          <TableCell>{getStatusBadge(emp.status)}</TableCell>
                          <TableCell className="text-right">
                            <div className="flex gap-2 justify-end">
                              <Button
                                size="sm"
                                variant="outline"
                                onClick={() => handleViewEmployee(emp)}
                              >
                                <Eye className="h-4 w-4" />
                              </Button>
                              <Button
                                size="sm"
                                variant="outline"
                                onClick={() => handleEditEmployee(emp)}
                              >
                                <Edit className="h-4 w-4" />
                              </Button>
                              {emp.status === 'active' && (
                                <Button
                                  size="sm"
                                  variant="outline"
                                  className="text-red-600"
                                  onClick={() => handleOpenTerminate(emp)}
                                >
                                  <UserX className="h-4 w-4" />
                                </Button>
                              )}
                            </div>
                          </TableCell>
                        </TableRow>
                      ))}
                    </TableBody>
                  </Table>
                </div>
              )}
            </CardContent>
          </Card>
        </TabsContent>

        {/* ATTENDANCE TAB */}
        <TabsContent value="attendance">
          <Card>
            <CardHeader>
              <div className="flex items-center justify-between">
                <CardTitle>Pointage Journalier</CardTitle>
                <Button
                  className="bg-indigo-600 hover:bg-indigo-700"
                  onClick={() => {
                    fetchAttendances();
                  }}
                >
                  <Clock className="h-4 w-4 mr-2" />
                  Actualiser
                </Button>
              </div>
            </CardHeader>
            <CardContent>
              <div className="text-center py-8 text-gray-500">
                Module de pointage : Fonctionnalité disponible prochainement.
                Les employés pourront pointer leur arrivée et départ, et les managers pourront consulter les présences en temps réel.
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        {/* LEAVES TAB */}
        <TabsContent value="leaves">
          <Card>
            <CardHeader>
              <div className="flex items-center justify-between">
                <CardTitle>Demandes de Congés</CardTitle>
                <Button
                  className="bg-indigo-600 hover:bg-indigo-700"
                  onClick={() => setShowLeaveRequestDialog(true)}
                >
                  <Plus className="h-4 w-4 mr-2" />
                  Nouvelle Demande
                </Button>
              </div>
            </CardHeader>
            <CardContent>
              {leaveRequests.length === 0 ? (
                <div className="text-center py-8 text-gray-500">Aucune demande en attente</div>
              ) : (
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead>Employé</TableHead>
                      <TableHead>Type</TableHead>
                      <TableHead>Du</TableHead>
                      <TableHead>Au</TableHead>
                      <TableHead>Jours</TableHead>
                      <TableHead>Raison</TableHead>
                      <TableHead>Actions</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {leaveRequests.map((req) => (
                      <TableRow key={req.id}>
                        <TableCell className="font-medium">{req.employee_name}</TableCell>
                        <TableCell>
                          <Badge variant="outline">{getLeaveTypeBadge(req.leave_type)}</Badge>
                        </TableCell>
                        <TableCell>{new Date(req.start_date).toLocaleDateString('fr-FR')}</TableCell>
                        <TableCell>{new Date(req.end_date).toLocaleDateString('fr-FR')}</TableCell>
                        <TableCell>{req.days_count}</TableCell>
                        <TableCell className="max-w-xs truncate">{req.reason || '-'}</TableCell>
                        <TableCell>
                          <div className="flex gap-2">
                            <Button 
                              size="sm" 
                              variant="outline"
                              className="text-green-600 hover:text-green-700 hover:border-green-600"
                              onClick={() => handleApproveLeave(req.id)}
                            >
                              <CheckCircle className="h-4 w-4 mr-1" />
                              Approuver
                            </Button>
                            <Button 
                              size="sm" 
                              variant="outline"
                              className="text-red-600 hover:text-red-700 hover:border-red-600"
                              onClick={() => handleRejectLeave(req.id)}
                            >
                              <XCircle className="h-4 w-4 mr-1" />
                              Rejeter
                            </Button>
                          </div>
                        </TableCell>
                      </TableRow>
                    ))}
                  </TableBody>
                </Table>
              )}
            </CardContent>
          </Card>
        </TabsContent>

        {/* PAYROLL TAB */}
        <TabsContent value="payroll">
          <Card>
            <CardHeader>
              <div className="flex items-center justify-between">
                <CardTitle>Gestion de la Paie</CardTitle>
                <Button
                  className="bg-indigo-600 hover:bg-indigo-700"
                  onClick={() => {
                    setShowGeneratePayslipDialog(true);
                    fetchPayslips();
                  }}
                >
                  <FileText className="h-4 w-4 mr-2" />
                  Générer Bulletin
                </Button>
              </div>
            </CardHeader>
            <CardContent>
              {payslips.length === 0 ? (
                <div className="text-center py-8 text-gray-500">
                  Aucun bulletin de paie. Cliquez sur "Générer Bulletin" pour créer.
                </div>
              ) : (
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead>Matricule</TableHead>
                      <TableHead>Employé</TableHead>
                      <TableHead>Période</TableHead>
                      <TableHead>Salaire Brut</TableHead>
                      <TableHead>Déductions</TableHead>
                      <TableHead>Salaire Net</TableHead>
                      <TableHead>Statut</TableHead>
                      <TableHead>Actions</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {payslips.map((payslip) => (
                      <TableRow key={payslip.id}>
                        <TableCell className="font-medium">{payslip.employee_number}</TableCell>
                        <TableCell>{payslip.employee_name}</TableCell>
                        <TableCell>
                          {getMonthName(payslip.period_month)} {payslip.period_year}
                        </TableCell>
                        <TableCell>{payslip.gross_salary.toLocaleString('fr-FR')} DA</TableCell>
                        <TableCell className="text-red-600">
                          -{payslip.total_deductions.toLocaleString('fr-FR')} DA
                        </TableCell>
                        <TableCell className="font-bold text-green-600">
                          {payslip.net_salary.toLocaleString('fr-FR')} DA
                        </TableCell>
                        <TableCell>
                          {payslip.is_paid ? (
                            <Badge className="bg-green-100 text-green-800">Payé</Badge>
                          ) : (
                            <Badge className="bg-orange-100 text-orange-800">En attente</Badge>
                          )}
                        </TableCell>
                        <TableCell>
                          <Button
                            size="sm"
                            variant="outline"
                            onClick={() => handleViewPayslip(payslip)}
                          >
                            <Eye className="h-4 w-4 mr-1" />
                            Voir
                          </Button>
                        </TableCell>
                      </TableRow>
                    ))}
                  </TableBody>
                </Table>
              )}
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>

      {/* CREATE/EDIT EMPLOYEE DIALOG */}
      <Dialog open={showEmployeeDialog || showEditEmployeeDialog} onOpenChange={(open) => {
        setShowEmployeeDialog(open);
        setShowEditEmployeeDialog(open);
        if (!open) {
          resetEmployeeForm();
          setSelectedEmployee(null);
        }
      }}>
        <DialogContent className="max-w-3xl max-h-[90vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle>{showEditEmployeeDialog ? 'Modifier Employé' : 'Nouvel Employé'}</DialogTitle>
            <DialogDescription>
              {showEditEmployeeDialog ? 'Modifier les informations de l\'employé' : 'Créer un nouveau dossier employé'}
            </DialogDescription>
          </DialogHeader>
          <div className="grid gap-4 py-4">
            {/* Informations personnelles */}
            <div>
              <h3 className="font-semibold mb-3 text-indigo-600">Informations Personnelles</h3>
              <div className="grid grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label>Prénom *</Label>
                  <Input
                    value={employeeForm.first_name}
                    onChange={(e) => setEmployeeForm({...employeeForm, first_name: e.target.value})}
                    placeholder="Prénom"
                  />
                </div>
                <div className="space-y-2">
                  <Label>Nom *</Label>
                  <Input
                    value={employeeForm.last_name}
                    onChange={(e) => setEmployeeForm({...employeeForm, last_name: e.target.value})}
                    placeholder="Nom"
                  />
                </div>
                <div className="space-y-2">
                  <Label>Date de Naissance *</Label>
                  <Input
                    type="date"
                    value={employeeForm.date_of_birth}
                    onChange={(e) => setEmployeeForm({...employeeForm, date_of_birth: e.target.value})}
                  />
                </div>
                <div className="space-y-2">
                  <Label>Lieu de Naissance</Label>
                  <Input
                    value={employeeForm.place_of_birth}
                    onChange={(e) => setEmployeeForm({...employeeForm, place_of_birth: e.target.value})}
                    placeholder="Lieu de naissance"
                  />
                </div>
                <div className="space-y-2">
                  <Label>Genre *</Label>
                  <Select
                    value={employeeForm.gender}
                    onValueChange={(value) => setEmployeeForm({...employeeForm, gender: value})}
                  >
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="M">Masculin</SelectItem>
                      <SelectItem value="F">Féminin</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
                <div className="space-y-2">
                  <Label>CIN *</Label>
                  <Input
                    value={employeeForm.cin_number}
                    onChange={(e) => setEmployeeForm({...employeeForm, cin_number: e.target.value})}
                    placeholder="N° Carte Identité"
                    disabled={showEditEmployeeDialog}
                  />
                </div>
                <div className="space-y-2">
                  <Label>N° Sécurité Sociale *</Label>
                  <Input
                    value={employeeForm.social_security_number}
                    onChange={(e) => setEmployeeForm({...employeeForm, social_security_number: e.target.value})}
                    placeholder="N° SS"
                  />
                </div>
                <div className="space-y-2">
                  <Label>Téléphone *</Label>
                  <Input
                    value={employeeForm.phone}
                    onChange={(e) => setEmployeeForm({...employeeForm, phone: e.target.value})}
                    placeholder="05XX XX XX XX"
                  />
                </div>
                <div className="space-y-2">
                  <Label>Email</Label>
                  <Input
                    type="email"
                    value={employeeForm.email}
                    onChange={(e) => setEmployeeForm({...employeeForm, email: e.target.value})}
                    placeholder="email@example.com"
                  />
                </div>
                <div className="space-y-2 col-span-2">
                  <Label>Adresse *</Label>
                  <Input
                    value={employeeForm.address}
                    onChange={(e) => setEmployeeForm({...employeeForm, address: e.target.value})}
                    placeholder="Adresse complète"
                  />
                </div>
                <div className="space-y-2">
                  <Label>Ville *</Label>
                  <Input
                    value={employeeForm.city}
                    onChange={(e) => setEmployeeForm({...employeeForm, city: e.target.value})}
                    placeholder="Ville"
                  />
                </div>
              </div>
            </div>

            {/* Informations emploi */}
            <div>
              <h3 className="font-semibold mb-3 text-indigo-600">Informations Emploi</h3>
              <div className="grid grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label>Matricule *</Label>
                  <Input
                    value={employeeForm.employee_number}
                    onChange={(e) => setEmployeeForm({...employeeForm, employee_number: e.target.value})}
                    placeholder="Ex: EMP001"
                    disabled={showEditEmployeeDialog}
                  />
                </div>
                <div className="space-y-2">
                  <Label>Date d'Embauche *</Label>
                  <Input
                    type="date"
                    value={employeeForm.hire_date}
                    onChange={(e) => setEmployeeForm({...employeeForm, hire_date: e.target.value})}
                  />
                </div>
                <div className="space-y-2">
                  <Label>Type de Contrat *</Label>
                  <Select
                    value={employeeForm.contract_type}
                    onValueChange={(value) => setEmployeeForm({...employeeForm, contract_type: value})}
                  >
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="cdi">CDI</SelectItem>
                      <SelectItem value="cdd">CDD</SelectItem>
                      <SelectItem value="daip">DAIP</SelectItem>
                      <SelectItem value="internship">Stage</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
                <div className="space-y-2">
                  <Label>Poste *</Label>
                  <Input
                    value={employeeForm.position}
                    onChange={(e) => setEmployeeForm({...employeeForm, position: e.target.value})}
                    placeholder="Ex: Mécanicien, Vendeur..."
                  />
                </div>
                <div className="space-y-2">
                  <Label>Département *</Label>
                  <Input
                    value={employeeForm.department}
                    onChange={(e) => setEmployeeForm({...employeeForm, department: e.target.value})}
                    placeholder="Ex: Atelier, Commercial..."
                  />
                </div>
                <div className="space-y-2">
                  <Label>Salaire de Base (DA) *</Label>
                  <Input
                    type="number"
                    value={employeeForm.base_salary}
                    onChange={(e) => setEmployeeForm({...employeeForm, base_salary: parseFloat(e.target.value) || 0})}
                    placeholder="Ex: 30000"
                  />
                </div>
              </div>
            </div>
          </div>
          <DialogFooter>
            <Button variant="outline" onClick={() => {
              setShowEmployeeDialog(false);
              setShowEditEmployeeDialog(false);
              resetEmployeeForm();
              setSelectedEmployee(null);
            }}>
              Annuler
            </Button>
            <Button
              className="bg-indigo-600 hover:bg-indigo-700"
              onClick={showEditEmployeeDialog ? handleUpdateEmployee : handleCreateEmployee}
              disabled={
                !employeeForm.first_name || !employeeForm.last_name || 
                !employeeForm.cin_number || !employeeForm.social_security_number ||
                !employeeForm.phone || !employeeForm.employee_number
              }
            >
              {showEditEmployeeDialog ? 'Modifier' : 'Créer Employé'}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* EMPLOYEE DETAILS DIALOG */}
      <Dialog open={showEmployeeDetailsDialog} onOpenChange={setShowEmployeeDetailsDialog}>
        <DialogContent className="max-w-3xl max-h-[90vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle>Détails Employé</DialogTitle>
          </DialogHeader>
          {selectedEmployee && (
            <div className="space-y-6">
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <h4 className="font-semibold text-sm text-gray-500">Matricule</h4>
                  <p className="text-lg font-medium">{selectedEmployee.employee_number}</p>
                </div>
                <div>
                  <h4 className="font-semibold text-sm text-gray-500">Nom Complet</h4>
                  <p className="text-lg font-medium">{selectedEmployee.first_name} {selectedEmployee.last_name}</p>
                </div>
                <div>
                  <h4 className="font-semibold text-sm text-gray-500">CIN</h4>
                  <p>{selectedEmployee.cin_number}</p>
                </div>
                <div>
                  <h4 className="font-semibold text-sm text-gray-500">N° Sécurité Sociale</h4>
                  <p>{selectedEmployee.social_security_number}</p>
                </div>
                <div>
                  <h4 className="font-semibold text-sm text-gray-500">Date de Naissance</h4>
                  <p>{new Date(selectedEmployee.date_of_birth).toLocaleDateString('fr-FR')}</p>
                </div>
                <div>
                  <h4 className="font-semibold text-sm text-gray-500">Téléphone</h4>
                  <p>{selectedEmployee.phone}</p>
                </div>
                <div className="col-span-2">
                  <h4 className="font-semibold text-sm text-gray-500">Adresse</h4>
                  <p>{selectedEmployee.address}, {selectedEmployee.city}</p>
                </div>
                <div>
                  <h4 className="font-semibold text-sm text-gray-500">Poste</h4>
                  <p className="font-medium">{selectedEmployee.position}</p>
                </div>
                <div>
                  <h4 className="font-semibold text-sm text-gray-500">Service</h4>
                  <p className="font-medium">{selectedEmployee.department}</p>
                </div>
                <div>
                  <h4 className="font-semibold text-sm text-gray-500">Type de Contrat</h4>
                  <p>{getContractTypeLabel(selectedEmployee.contract_type)}</p>
                </div>
                <div>
                  <h4 className="font-semibold text-sm text-gray-500">Date d'Embauche</h4>
                  <p>{new Date(selectedEmployee.hire_date).toLocaleDateString('fr-FR')}</p>
                </div>
                <div>
                  <h4 className="font-semibold text-sm text-gray-500">Salaire de Base</h4>
                  <p className="text-lg font-bold text-green-600">
                    {selectedEmployee.base_salary.toLocaleString('fr-FR')} DA
                  </p>
                </div>
                <div>
                  <h4 className="font-semibold text-sm text-gray-500">Statut</h4>
                  <div className="mt-1">{getStatusBadge(selectedEmployee.status)}</div>
                </div>
              </div>
            </div>
          )}
          <DialogFooter>
            <Button variant="outline" onClick={() => {
              setShowEmployeeDetailsDialog(false);
              setSelectedEmployee(null);
            }}>
              Fermer
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* TERMINATE EMPLOYEE DIALOG */}
      <Dialog open={showTerminateDialog} onOpenChange={setShowTerminateDialog}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Résiliation de Contrat</DialogTitle>
            <DialogDescription>
              {selectedEmployee && `Fin de contrat pour ${selectedEmployee.first_name} ${selectedEmployee.last_name}`}
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div className="space-y-2">
              <Label>Date de Fin de Contrat *</Label>
              <Input
                type="date"
                value={terminateForm.termination_date}
                onChange={(e) => setTerminateForm({...terminateForm, termination_date: e.target.value})}
              />
            </div>
            <div className="space-y-2">
              <Label>Raison de la Résiliation *</Label>
              <Textarea
                value={terminateForm.termination_reason}
                onChange={(e) => setTerminateForm({...terminateForm, termination_reason: e.target.value})}
                placeholder="Ex: Démission, Fin de contrat, Licenciement..."
                rows={4}
              />
            </div>
          </div>
          <DialogFooter>
            <Button variant="outline" onClick={() => {
              setShowTerminateDialog(false);
              resetTerminateForm();
              setSelectedEmployee(null);
            }}>
              Annuler
            </Button>
            <Button
              className="bg-red-600 hover:bg-red-700"
              onClick={handleTerminateEmployee}
              disabled={!terminateForm.termination_reason}
            >
              Confirmer Résiliation
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* LEAVE REQUEST DIALOG */}
      <Dialog open={showLeaveRequestDialog} onOpenChange={setShowLeaveRequestDialog}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Nouvelle Demande de Congé</DialogTitle>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div className="space-y-2">
              <Label>Employé *</Label>
              <Select
                value={leaveRequestForm.employee_id}
                onValueChange={(value) => setLeaveRequestForm({...leaveRequestForm, employee_id: value})}
              >
                <SelectTrigger>
                  <SelectValue placeholder="Sélectionner un employé" />
                </SelectTrigger>
                <SelectContent>
                  {employees.map((emp) => (
                    <SelectItem key={emp.id} value={emp.id}>
                      {emp.first_name} {emp.last_name} - {emp.employee_number}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            <div className="space-y-2">
              <Label>Type de Congé *</Label>
              <Select
                value={leaveRequestForm.leave_type}
                onValueChange={(value) => setLeaveRequestForm({...leaveRequestForm, leave_type: value})}
              >
                <SelectTrigger>
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="annual">Congé annuel</SelectItem>
                  <SelectItem value="sick">Maladie</SelectItem>
                  <SelectItem value="maternity">Maternité</SelectItem>
                  <SelectItem value="paternity">Paternité</SelectItem>
                  <SelectItem value="unpaid">Sans solde</SelectItem>
                  <SelectItem value="exceptional">Exceptionnel</SelectItem>
                </SelectContent>
              </Select>
            </div>
            <div className="grid grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label>Date de Début *</Label>
                <Input
                  type="date"
                  value={leaveRequestForm.start_date}
                  onChange={(e) => setLeaveRequestForm({...leaveRequestForm, start_date: e.target.value})}
                />
              </div>
              <div className="space-y-2">
                <Label>Date de Fin *</Label>
                <Input
                  type="date"
                  value={leaveRequestForm.end_date}
                  onChange={(e) => setLeaveRequestForm({...leaveRequestForm, end_date: e.target.value})}
                />
              </div>
            </div>
            <div className="space-y-2">
              <Label>Raison</Label>
              <Textarea
                value={leaveRequestForm.reason}
                onChange={(e) => setLeaveRequestForm({...leaveRequestForm, reason: e.target.value})}
                placeholder="Motif de la demande..."
                rows={3}
              />
            </div>
          </div>
          <DialogFooter>
            <Button variant="outline" onClick={() => {
              setShowLeaveRequestDialog(false);
              resetLeaveRequestForm();
            }}>
              Annuler
            </Button>
            <Button
              className="bg-indigo-600 hover:bg-indigo-700"
              onClick={handleCreateLeaveRequest}
              disabled={!leaveRequestForm.employee_id || !leaveRequestForm.start_date || !leaveRequestForm.end_date}
            >
              Créer Demande
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* GENERATE PAYSLIP DIALOG */}
      <Dialog open={showGeneratePayslipDialog} onOpenChange={setShowGeneratePayslipDialog}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Générer Bulletin de Paie</DialogTitle>
            <DialogDescription>
              Créer un bulletin de paie pour un employé
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div className="space-y-2">
              <Label>Employé *</Label>
              <Select
                value={generatePayslipForm.employee_id}
                onValueChange={(value) => setGeneratePayslipForm({...generatePayslipForm, employee_id: value})}
              >
                <SelectTrigger>
                  <SelectValue placeholder="Sélectionner un employé" />
                </SelectTrigger>
                <SelectContent>
                  {employees.filter(e => e.status === 'active').map((emp) => (
                    <SelectItem key={emp.id} value={emp.id}>
                      {emp.first_name} {emp.last_name} - {emp.employee_number}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            <div className="grid grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label>Mois *</Label>
                <Select
                  value={generatePayslipForm.period_month.toString()}
                  onValueChange={(value) => setGeneratePayslipForm({...generatePayslipForm, period_month: parseInt(value)})}
                >
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    {Array.from({length: 12}, (_, i) => i + 1).map(month => (
                      <SelectItem key={month} value={month.toString()}>
                        {getMonthName(month)}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-2">
                <Label>Année *</Label>
                <Select
                  value={generatePayslipForm.period_year.toString()}
                  onValueChange={(value) => setGeneratePayslipForm({...generatePayslipForm, period_year: parseInt(value)})}
                >
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    {Array.from({length: 5}, (_, i) => new Date().getFullYear() - i).map(year => (
                      <SelectItem key={year} value={year.toString()}>
                        {year}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
            </div>
          </div>
          <DialogFooter>
            <Button variant="outline" onClick={() => {
              setShowGeneratePayslipDialog(false);
              resetGeneratePayslipForm();
            }}>
              Annuler
            </Button>
            <Button
              className="bg-indigo-600 hover:bg-indigo-700"
              onClick={handleGeneratePayslip}
              disabled={!generatePayslipForm.employee_id}
            >
              Générer
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* PAYSLIP DETAILS DIALOG */}
      <Dialog open={showPayslipDetailsDialog} onOpenChange={setShowPayslipDetailsDialog}>
        <DialogContent className="max-w-2xl">
          <DialogHeader>
            <DialogTitle>Bulletin de Paie</DialogTitle>
            <DialogDescription>
              {selectedPayslip && `${getMonthName(selectedPayslip.period_month)} ${selectedPayslip.period_year}`}
            </DialogDescription>
          </DialogHeader>
          {selectedPayslip && (
            <div className="space-y-6">
              {/* Employee Info */}
              <div className="bg-gray-50 p-4 rounded-lg">
                <div className="grid grid-cols-2 gap-4">
                  <div>
                    <p className="text-sm text-gray-500">Matricule</p>
                    <p className="font-medium">{selectedPayslip.employee_number}</p>
                  </div>
                  <div>
                    <p className="text-sm text-gray-500">Nom et Prénom</p>
                    <p className="font-medium">{selectedPayslip.employee_name}</p>
                  </div>
                </div>
              </div>

              {/* Salary Details */}
              <div>
                <h4 className="font-semibold mb-3">Détails de la Rémunération</h4>
                <div className="space-y-2">
                  <div className="flex justify-between py-2 border-b">
                    <span>Salaire de Base</span>
                    <span className="font-medium">{selectedPayslip.base_salary.toLocaleString('fr-FR')} DA</span>
                  </div>
                  {selectedPayslip.allowances && selectedPayslip.allowances.length > 0 && (
                    <>
                      {selectedPayslip.allowances.map((item, idx) => (
                        <div key={idx} className="flex justify-between py-2 border-b">
                          <span>{item.description}</span>
                          <span className="font-medium text-green-600">+{item.amount.toLocaleString('fr-FR')} DA</span>
                        </div>
                      ))}
                    </>
                  )}
                  <div className="flex justify-between py-2 border-b font-semibold bg-green-50 px-2">
                    <span>Salaire Brut</span>
                    <span>{selectedPayslip.gross_salary.toLocaleString('fr-FR')} DA</span>
                  </div>
                </div>
              </div>

              {/* Deductions */}
              <div>
                <h4 className="font-semibold mb-3">Charges et Déductions</h4>
                <div className="space-y-2">
                  <div className="flex justify-between py-2 border-b">
                    <span>CNSS (9%)</span>
                    <span className="font-medium text-red-600">-{selectedPayslip.cnss_employee.toLocaleString('fr-FR')} DA</span>
                  </div>
                  <div className="flex justify-between py-2 border-b">
                    <span>IRG</span>
                    <span className="font-medium text-red-600">-{selectedPayslip.irg_amount.toLocaleString('fr-FR')} DA</span>
                  </div>
                  <div className="flex justify-between py-2 border-b font-semibold bg-red-50 px-2">
                    <span>Total Déductions</span>
                    <span>-{selectedPayslip.total_deductions.toLocaleString('fr-FR')} DA</span>
                  </div>
                </div>
              </div>

              {/* Net Salary */}
              <div className="bg-indigo-50 p-4 rounded-lg">
                <div className="flex justify-between items-center">
                  <span className="text-lg font-semibold">Salaire Net à Payer</span>
                  <span className="text-2xl font-bold text-indigo-600">
                    {selectedPayslip.net_salary.toLocaleString('fr-FR')} DA
                  </span>
                </div>
              </div>

              {/* Employer Charges (info) */}
              <div className="bg-gray-50 p-4 rounded-lg">
                <p className="text-sm text-gray-600 mb-2">Charges Patronales (informatives)</p>
                <div className="flex justify-between">
                  <span className="text-sm">CNSS Employeur (26%)</span>
                  <span className="font-medium">{selectedPayslip.cnss_employer.toLocaleString('fr-FR')} DA</span>
                </div>
              </div>

              {/* Payment Info */}
              <div className="text-sm text-gray-500">
                <p>Date de paiement: {new Date(selectedPayslip.payment_date).toLocaleDateString('fr-FR')}</p>
                {selectedPayslip.payment_reference && (
                  <p>Référence: {selectedPayslip.payment_reference}</p>
                )}
              </div>
            </div>
          )}
          <DialogFooter>
            <Button variant="outline" onClick={() => {
              setShowPayslipDetailsDialog(false);
              setSelectedPayslip(null);
            }}>
              Fermer
            </Button>
            <Button className="bg-indigo-600 hover:bg-indigo-700">
              <FileText className="h-4 w-4 mr-2" />
              Télécharger PDF
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
};

export default HRPage;
