import React, { useState, useEffect } from 'react';
import { useTranslation } from 'react-i18next';
import api from '@/services/api';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Badge } from '@/components/ui/badge';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { DollarSign, TrendingUp, Wallet, Building2, Plus } from 'lucide-react';

const FinancePage = () => {
  const { t } = useTranslation();
  const [loading, setLoading] = useState(true);
  const [activeTab, setActiveTab] = useState('dashboard');
  
  // Dashboard stats
  const [dashboardStats, setDashboardStats] = useState({
    total_bank_balance: 0,
    total_cash_balance: 0,
    total_treasury: 0,
    bank_accounts_count: 0,
    cash_registers_count: 0
  });
  
  // Bank accounts
  const [bankAccounts, setBankAccounts] = useState([]);
  const [cashRegisters, setCashRegisters] = useState([]);
  const [journalEntries, setJournalEntries] = useState([]);

  useEffect(() => {
    fetchDashboard();
    fetchBankAccounts();
    fetchCashRegisters();
  }, []);

  const fetchDashboard = async () => {
    try {
      setLoading(true);
      const response = await api.get('/finance/dashboard');
      setDashboardStats(response.data);
    } catch (error) {
      console.error('Error fetching finance dashboard:', error);
    } finally {
      setLoading(false);
    }
  };

  const fetchBankAccounts = async () => {
    try {
      const response = await api.get('/finance/banks');
      setBankAccounts(response.data);
    } catch (error) {
      console.error('Error fetching bank accounts:', error);
    }
  };

  const fetchCashRegisters = async () => {
    try {
      const response = await api.get('/finance/cash-registers');
      setCashRegisters(response.data);
    } catch (error) {
      console.error('Error fetching cash registers:', error);
    }
  };

  const formatCurrency = (amount) => {
    return new Intl.NumberFormat('fr-DZ', {
      style: 'currency',
      currency: 'DZD',
      minimumFractionDigits: 0
    }).format(amount);
  };

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold text-gray-900">Finance & Comptabilité</h1>
          <p className="text-gray-600 mt-2">Gestion comptable et trésorerie</p>
        </div>
      </div>

      {/* Stats Cards */}
      <div className="grid gap-6 md:grid-cols-4">
        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Trésorerie Totale</CardTitle>
            <DollarSign className="h-4 w-4 text-green-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-green-600">
              {formatCurrency(dashboardStats.total_treasury)}
            </div>
            <p className="text-xs text-gray-500 mt-1">
              Banques + Caisses
            </p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Solde Banques</CardTitle>
            <Building2 className="h-4 w-4 text-blue-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">
              {formatCurrency(dashboardStats.total_bank_balance)}
            </div>
            <p className="text-xs text-gray-500 mt-1">
              {dashboardStats.bank_accounts_count} compte(s)
            </p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Solde Caisses</CardTitle>
            <Wallet className="h-4 w-4 text-purple-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">
              {formatCurrency(dashboardStats.total_cash_balance)}
            </div>
            <p className="text-xs text-gray-500 mt-1">
              {dashboardStats.cash_registers_count} caisse(s)
            </p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Créances Clients</CardTitle>
            <TrendingUp className="h-4 w-4 text-orange-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">
              {formatCurrency(0)}
            </div>
            <p className="text-xs text-gray-500 mt-1">
              À encaisser
            </p>
          </CardContent>
        </Card>
      </div>

      {/* Tabs */}
      <Tabs value={activeTab} onValueChange={setActiveTab}>
        <TabsList>
          <TabsTrigger value="dashboard">Vue d'ensemble</TabsTrigger>
          <TabsTrigger value="banks">Banques</TabsTrigger>
          <TabsTrigger value="cash">Caisses</TabsTrigger>
          <TabsTrigger value="journal">Écritures</TabsTrigger>
        </TabsList>

        <TabsContent value="dashboard" className="space-y-4">
          <div className="grid gap-6 md:grid-cols-2">
            {/* Recent Bank Accounts */}
            <Card>
              <CardHeader>
                <CardTitle>Comptes Bancaires</CardTitle>
                <CardDescription>Liste des comptes actifs</CardDescription>
              </CardHeader>
              <CardContent>
                {bankAccounts.length === 0 ? (
                  <p className="text-center text-gray-500 py-4">Aucun compte bancaire</p>
                ) : (
                  <div className="space-y-3">
                    {bankAccounts.slice(0, 5).map((bank) => (
                      <div key={bank.id} className="flex items-center justify-between p-3 border rounded-lg">
                        <div>
                          <p className="font-medium">{bank.bank_name}</p>
                          <p className="text-sm text-gray-500">{bank.account_number}</p>
                        </div>
                        <div className="text-right">
                          <p className="font-bold text-green-600">{formatCurrency(bank.balance)}</p>
                          <Badge variant={bank.is_active ? "success" : "destructive"}>
                            {bank.is_active ? "Actif" : "Inactif"}
                          </Badge>
                        </div>
                      </div>
                    ))}
                  </div>
                )}
              </CardContent>
            </Card>

            {/* Cash Registers */}
            <Card>
              <CardHeader>
                <CardTitle>Caisses</CardTitle>
                <CardDescription>États des caisses</CardDescription>
              </CardHeader>
              <CardContent>
                {cashRegisters.length === 0 ? (
                  <p className="text-center text-gray-500 py-4">Aucune caisse</p>
                ) : (
                  <div className="space-y-3">
                    {cashRegisters.map((cash) => (
                      <div key={cash.id} className="flex items-center justify-between p-3 border rounded-lg">
                        <div>
                          <p className="font-medium">{cash.name}</p>
                          <p className="text-sm text-gray-500">Resp: {cash.responsible}</p>
                        </div>
                        <div className="text-right">
                          <p className="font-bold text-purple-600">{formatCurrency(cash.balance)}</p>
                          <Badge variant={cash.is_active ? "success" : "destructive"}>
                            {cash.is_active ? "Active" : "Inactive"}
                          </Badge>
                        </div>
                      </div>
                    ))}
                  </div>
                )}
              </CardContent>
            </Card>
          </div>
        </TabsContent>

        <TabsContent value="banks" className="space-y-4">
          <Card>
            <CardHeader>
              <div className="flex items-center justify-between">
                <div>
                  <CardTitle>Comptes Bancaires</CardTitle>
                  <CardDescription>Gestion des comptes en banque</CardDescription>
                </div>
                <Button className="bg-indigo-600">
                  <Plus className="h-4 w-4 mr-2" />
                  Nouveau Compte
                </Button>
              </div>
            </CardHeader>
            <CardContent>
              {bankAccounts.length === 0 ? (
                <div className="text-center py-8 text-gray-500">
                  Aucun compte bancaire enregistré
                </div>
              ) : (
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead>Banque</TableHead>
                      <TableHead>N° Compte</TableHead>
                      <TableHead>RIB</TableHead>
                      <TableHead className="text-right">Solde</TableHead>
                      <TableHead>Statut</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {bankAccounts.map((bank) => (
                      <TableRow key={bank.id}>
                        <TableCell className="font-medium">{bank.bank_name}</TableCell>
                        <TableCell>{bank.account_number}</TableCell>
                        <TableCell className="font-mono text-sm">{bank.rib}</TableCell>
                        <TableCell className="text-right font-bold text-green-600">
                          {formatCurrency(bank.balance)}
                        </TableCell>
                        <TableCell>
                          <Badge variant={bank.is_active ? "success" : "destructive"}>
                            {bank.is_active ? "Actif" : "Inactif"}
                          </Badge>
                        </TableCell>
                      </TableRow>
                    ))}
                  </TableBody>
                </Table>
              )}
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="cash" className="space-y-4">
          <Card>
            <CardHeader>
              <div className="flex items-center justify-between">
                <div>
                  <CardTitle>Caisses</CardTitle>
                  <CardDescription>Gestion des caisses espèces</CardDescription>
                </div>
                <Button className="bg-indigo-600">
                  <Plus className="h-4 w-4 mr-2" />
                  Nouvelle Caisse
                </Button>
              </div>
            </CardHeader>
            <CardContent>
              {cashRegisters.length === 0 ? (
                <div className="text-center py-8 text-gray-500">
                  Aucune caisse enregistrée
                </div>
              ) : (
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead>Nom</TableHead>
                      <TableHead>Responsable</TableHead>
                      <TableHead>Site</TableHead>
                      <TableHead className="text-right">Solde</TableHead>
                      <TableHead>Statut</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {cashRegisters.map((cash) => (
                      <TableRow key={cash.id}>
                        <TableCell className="font-medium">{cash.name}</TableCell>
                        <TableCell>{cash.responsible}</TableCell>
                        <TableCell>{cash.site_id || 'Principal'}</TableCell>
                        <TableCell className="text-right font-bold text-purple-600">
                          {formatCurrency(cash.balance)}
                        </TableCell>
                        <TableCell>
                          <Badge variant={cash.is_active ? "success" : "destructive"}>
                            {cash.is_active ? "Active" : "Inactive"}
                          </Badge>
                        </TableCell>
                      </TableRow>
                    ))}
                  </TableBody>
                </Table>
              )}
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="journal" className="space-y-4">
          <Card>
            <CardHeader>
              <div className="flex items-center justify-between">
                <div>
                  <CardTitle>Écritures Comptables</CardTitle>
                  <CardDescription>Journal des opérations</CardDescription>
                </div>
                <Button className="bg-indigo-600">
                  <Plus className="h-4 w-4 mr-2" />
                  Nouvelle Écriture
                </Button>
              </div>
            </CardHeader>
            <CardContent>
              <div className="text-center py-8 text-gray-500">
                Fonctionnalité en développement - Les écritures comptables seront affichées ici
              </div>
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>
    </div>
  );
};

export default FinancePage;
