import React, { useState, useEffect } from 'react';
import { useTranslation } from 'react-i18next';
import api from '@/services/api';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Badge } from '@/components/ui/badge';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle } from '@/components/ui/dialog';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { DollarSign, TrendingUp, Wallet, Building2, Plus, Edit, Trash2 } from 'lucide-react';

const FinancePage = () => {
  const { t } = useTranslation();
  const [loading, setLoading] = useState(true);
  const [activeTab, setActiveTab] = useState('dashboard');
  
  // Dashboard stats
  const [dashboardStats, setDashboardStats] = useState({
    total_bank_balance: 0,
    total_cash_balance: 0,
    total_treasury: 0,
    bank_accounts_count: 0,
    cash_registers_count: 0
  });
  
  // Data
  const [bankAccounts, setBankAccounts] = useState([]);
  const [cashRegisters, setCashRegisters] = useState([]);
  const [journalEntries, setJournalEntries] = useState([]);
  
  // Dialogs
  const [showBankDialog, setShowBankDialog] = useState(false);
  const [showCashDialog, setShowCashDialog] = useState(false);
  const [showJournalDialog, setShowJournalDialog] = useState(false);
  const [showJournalDetailsDialog, setShowJournalDetailsDialog] = useState(false);
  const [selectedEntry, setSelectedEntry] = useState(null);
  
  // Form data
  const [bankForm, setBankForm] = useState({
    bank_name: '',
    account_number: '',
    rib: '',
    iban: '',
    swift: '',
    currency: 'DZD',
    balance: 0
  });
  
  const [cashForm, setCashForm] = useState({
    name: '',
    responsible: '',
    site_id: '',
    balance: 0
  });

  const [journalForm, setJournalForm] = useState({
    journal_type: 'operations',
    date: new Date().toISOString().split('T')[0],
    description: '',
    reference: '',
    lines: [
      { account_code: '', account_name: '', debit: 0, credit: 0, description: '' },
      { account_code: '', account_name: '', debit: 0, credit: 0, description: '' }
    ]
  });

  useEffect(() => {
    fetchDashboard();
    fetchBankAccounts();
    fetchCashRegisters();
    if (activeTab === 'journal') {
      fetchJournalEntries();
    }
  }, [activeTab]);

  const fetchDashboard = async () => {
    try {
      setLoading(true);
      const response = await api.get('/finance/dashboard');
      setDashboardStats(response.data);
    } catch (error) {
      console.error('Error fetching finance dashboard:', error);
    } finally {
      setLoading(false);
    }
  };

  const fetchBankAccounts = async () => {
    try {
      const response = await api.get('/finance/banks');
      setBankAccounts(response.data);
    } catch (error) {
      console.error('Error fetching bank accounts:', error);
    }
  };

  const fetchCashRegisters = async () => {
    try {
      const response = await api.get('/finance/cash-registers');
      setCashRegisters(response.data);
    } catch (error) {
      console.error('Error fetching cash registers:', error);
    }
  };

  const handleCreateBank = async () => {
    try {
      await api.post('/finance/banks', bankForm);
      setShowBankDialog(false);
      resetBankForm();
      fetchBankAccounts();
      fetchDashboard();
      alert('Compte bancaire créé avec succès');
    } catch (error) {
      console.error('Error creating bank:', error);
      alert('Erreur lors de la création du compte bancaire');
    }
  };

  const handleCreateCash = async () => {
    try {
      await api.post('/finance/cash-registers', cashForm);
      setShowCashDialog(false);
      resetCashForm();
      fetchCashRegisters();
      fetchDashboard();
      alert('Caisse créée avec succès');
    } catch (error) {
      console.error('Error creating cash:', error);
      alert('Erreur lors de la création de la caisse');
    }
  };

  const resetBankForm = () => {
    setBankForm({
      bank_name: '',
      account_number: '',
      rib: '',
      iban: '',
      swift: '',
      currency: 'DZD',
      balance: 0
    });
  };

  const resetCashForm = () => {
    setCashForm({
      name: '',
      responsible: '',
      site_id: '',
      balance: 0
    });
  };

  const resetJournalForm = () => {
    setJournalForm({
      journal_type: 'operations',
      date: new Date().toISOString().split('T')[0],
      description: '',
      reference: '',
      lines: [
        { account_code: '', account_name: '', debit: 0, credit: 0, description: '' },
        { account_code: '', account_name: '', debit: 0, credit: 0, description: '' }
      ]
    });
  };

  const fetchJournalEntries = async () => {
    try {
      const response = await api.get('/finance/journal-entries');
      setJournalEntries(response.data);
    } catch (error) {
      console.error('Error fetching journal entries:', error);
    }
  };

  const handleCreateJournalEntry = async () => {
    try {
      // Validate balance
      const totalDebit = journalForm.lines.reduce((sum, line) => sum + (parseFloat(line.debit) || 0), 0);
      const totalCredit = journalForm.lines.reduce((sum, line) => sum + (parseFloat(line.credit) || 0), 0);
      
      if (Math.abs(totalDebit - totalCredit) > 0.01) {
        alert(`Écriture non équilibrée ! Débit: ${totalDebit} DA, Crédit: ${totalCredit} DA`);
        return;
      }

      await api.post('/finance/journal-entries', journalForm);
      setShowJournalDialog(false);
      resetJournalForm();
      fetchJournalEntries();
      alert('Écriture comptable créée avec succès');
    } catch (error) {
      console.error('Error creating journal entry:', error);
      alert(error.response?.data?.detail || 'Erreur lors de la création');
    }
  };

  const handleAddJournalLine = () => {
    setJournalForm({
      ...journalForm,
      lines: [...journalForm.lines, { account_code: '', account_name: '', debit: 0, credit: 0, description: '' }]
    });
  };

  const handleRemoveJournalLine = (index) => {
    if (journalForm.lines.length > 2) {
      setJournalForm({
        ...journalForm,
        lines: journalForm.lines.filter((_, i) => i !== index)
      });
    }
  };

  const handleJournalLineChange = (index, field, value) => {
    const newLines = [...journalForm.lines];
    newLines[index] = { ...newLines[index], [field]: value };
    setJournalForm({ ...journalForm, lines: newLines });
  };

  const getTotalDebit = () => {
    return journalForm.lines.reduce((sum, line) => sum + (parseFloat(line.debit) || 0), 0);
  };

  const getTotalCredit = () => {
    return journalForm.lines.reduce((sum, line) => sum + (parseFloat(line.credit) || 0), 0);
  };

  const getJournalTypeBadge = (type) => {
    const badges = {
      sales: { label: 'Ventes', class: 'bg-green-100 text-green-800' },
      purchases: { label: 'Achats', class: 'bg-blue-100 text-blue-800' },
      bank: { label: 'Banque', class: 'bg-purple-100 text-purple-800' },
      cash: { label: 'Caisse', class: 'bg-yellow-100 text-yellow-800' },
      operations: { label: 'Opérations Diverses', class: 'bg-gray-100 text-gray-800' }
    };
    const badge = badges[type] || badges.operations;
    return <Badge className={badge.class}>{badge.label}</Badge>;
  };

  const getStatusBadge = (status) => {
    const badges = {
      draft: { label: 'Brouillon', class: 'bg-gray-100 text-gray-800' },
      posted: { label: 'Comptabilisé', class: 'bg-blue-100 text-blue-800' },
      validated: { label: 'Validé', class: 'bg-green-100 text-green-800' },
      cancelled: { label: 'Annulé', class: 'bg-red-100 text-red-800' }
    };
    const badge = badges[status] || badges.draft;
    return <Badge className={badge.class}>{badge.label}</Badge>;
  };

  const formatCurrency = (amount) => {
    return new Intl.NumberFormat('fr-DZ', {
      style: 'currency',
      currency: 'DZD',
      minimumFractionDigits: 0
    }).format(amount);
  };

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold text-gray-900">Finance & Comptabilité</h1>
          <p className="text-gray-600 mt-2">Gestion comptable et trésorerie</p>
        </div>
      </div>

      {/* Stats Cards */}
      <div className="grid gap-6 md:grid-cols-4">
        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Trésorerie Totale</CardTitle>
            <DollarSign className="h-4 w-4 text-green-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-green-600">
              {formatCurrency(dashboardStats.total_treasury)}
            </div>
            <p className="text-xs text-gray-500 mt-1">Banques + Caisses</p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Solde Banques</CardTitle>
            <Building2 className="h-4 w-4 text-blue-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">
              {formatCurrency(dashboardStats.total_bank_balance)}
            </div>
            <p className="text-xs text-gray-500 mt-1">
              {dashboardStats.bank_accounts_count} compte(s)
            </p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Solde Caisses</CardTitle>
            <Wallet className="h-4 w-4 text-purple-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">
              {formatCurrency(dashboardStats.total_cash_balance)}
            </div>
            <p className="text-xs text-gray-500 mt-1">
              {dashboardStats.cash_registers_count} caisse(s)
            </p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Créances Clients</CardTitle>
            <TrendingUp className="h-4 w-4 text-orange-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{formatCurrency(0)}</div>
            <p className="text-xs text-gray-500 mt-1">À encaisser</p>
          </CardContent>
        </Card>
      </div>

      {/* Tabs */}
      <Tabs value={activeTab} onValueChange={setActiveTab}>
        <TabsList>
          <TabsTrigger value="dashboard">Vue d'ensemble</TabsTrigger>
          <TabsTrigger value="banks">Banques</TabsTrigger>
          <TabsTrigger value="cash">Caisses</TabsTrigger>
          <TabsTrigger value="journal">Écritures</TabsTrigger>
        </TabsList>

        <TabsContent value="dashboard" className="space-y-4">
          <div className="grid gap-6 md:grid-cols-2">
            {/* Recent Bank Accounts */}
            <Card>
              <CardHeader>
                <CardTitle>Comptes Bancaires</CardTitle>
                <CardDescription>Liste des comptes actifs</CardDescription>
              </CardHeader>
              <CardContent>
                {bankAccounts.length === 0 ? (
                  <p className="text-center text-gray-500 py-4">Aucun compte bancaire</p>
                ) : (
                  <div className="space-y-3">
                    {bankAccounts.slice(0, 5).map((bank) => (
                      <div key={bank.id} className="flex items-center justify-between p-3 border rounded-lg">
                        <div>
                          <p className="font-medium">{bank.bank_name}</p>
                          <p className="text-sm text-gray-500">{bank.account_number}</p>
                        </div>
                        <div className="text-right">
                          <p className="font-bold text-green-600">{formatCurrency(bank.balance)}</p>
                          <Badge variant={bank.is_active ? "success" : "destructive"}>
                            {bank.is_active ? "Actif" : "Inactif"}
                          </Badge>
                        </div>
                      </div>
                    ))}
                  </div>
                )}
              </CardContent>
            </Card>

            {/* Cash Registers */}
            <Card>
              <CardHeader>
                <CardTitle>Caisses</CardTitle>
                <CardDescription>États des caisses</CardDescription>
              </CardHeader>
              <CardContent>
                {cashRegisters.length === 0 ? (
                  <p className="text-center text-gray-500 py-4">Aucune caisse</p>
                ) : (
                  <div className="space-y-3">
                    {cashRegisters.map((cash) => (
                      <div key={cash.id} className="flex items-center justify-between p-3 border rounded-lg">
                        <div>
                          <p className="font-medium">{cash.name}</p>
                          <p className="text-sm text-gray-500">Resp: {cash.responsible}</p>
                        </div>
                        <div className="text-right">
                          <p className="font-bold text-purple-600">{formatCurrency(cash.balance)}</p>
                          <Badge variant={cash.is_active ? "success" : "destructive"}>
                            {cash.is_active ? "Active" : "Inactive"}
                          </Badge>
                        </div>
                      </div>
                    ))}
                  </div>
                )}
              </CardContent>
            </Card>
          </div>
        </TabsContent>

        <TabsContent value="banks" className="space-y-4">
          <Card>
            <CardHeader>
              <div className="flex items-center justify-between">
                <div>
                  <CardTitle>Comptes Bancaires</CardTitle>
                  <CardDescription>Gestion des comptes en banque</CardDescription>
                </div>
                <Button 
                  className="bg-indigo-600"
                  onClick={() => setShowBankDialog(true)}
                >
                  <Plus className="h-4 w-4 mr-2" />
                  Nouveau Compte
                </Button>
              </div>
            </CardHeader>
            <CardContent>
              {bankAccounts.length === 0 ? (
                <div className="text-center py-8 text-gray-500">
                  Aucun compte bancaire enregistré
                </div>
              ) : (
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead>Banque</TableHead>
                      <TableHead>N° Compte</TableHead>
                      <TableHead>RIB</TableHead>
                      <TableHead className="text-right">Solde</TableHead>
                      <TableHead>Statut</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {bankAccounts.map((bank) => (
                      <TableRow key={bank.id}>
                        <TableCell className="font-medium">{bank.bank_name}</TableCell>
                        <TableCell>{bank.account_number}</TableCell>
                        <TableCell className="font-mono text-sm">{bank.rib}</TableCell>
                        <TableCell className="text-right font-bold text-green-600">
                          {formatCurrency(bank.balance)}
                        </TableCell>
                        <TableCell>
                          <Badge variant={bank.is_active ? "success" : "destructive"}>
                            {bank.is_active ? "Actif" : "Inactif"}
                          </Badge>
                        </TableCell>
                      </TableRow>
                    ))}
                  </TableBody>
                </Table>
              )}
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="cash" className="space-y-4">
          <Card>
            <CardHeader>
              <div className="flex items-center justify-between">
                <div>
                  <CardTitle>Caisses</CardTitle>
                  <CardDescription>Gestion des caisses espèces</CardDescription>
                </div>
                <Button 
                  className="bg-indigo-600"
                  onClick={() => setShowCashDialog(true)}
                >
                  <Plus className="h-4 w-4 mr-2" />
                  Nouvelle Caisse
                </Button>
              </div>
            </CardHeader>
            <CardContent>
              {cashRegisters.length === 0 ? (
                <div className="text-center py-8 text-gray-500">
                  Aucune caisse enregistrée
                </div>
              ) : (
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead>Nom</TableHead>
                      <TableHead>Responsable</TableHead>
                      <TableHead>Site</TableHead>
                      <TableHead className="text-right">Solde</TableHead>
                      <TableHead>Statut</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {cashRegisters.map((cash) => (
                      <TableRow key={cash.id}>
                        <TableCell className="font-medium">{cash.name}</TableCell>
                        <TableCell>{cash.responsible}</TableCell>
                        <TableCell>{cash.site_id || 'Principal'}</TableCell>
                        <TableCell className="text-right font-bold text-purple-600">
                          {formatCurrency(cash.balance)}
                        </TableCell>
                        <TableCell>
                          <Badge variant={cash.is_active ? "success" : "destructive"}>
                            {cash.is_active ? "Active" : "Inactive"}
                          </Badge>
                        </TableCell>
                      </TableRow>
                    ))}
                  </TableBody>
                </Table>
              )}
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="journal">
          <Card>
            <CardHeader>
              <div className="flex items-center justify-between">
                <div>
                  <CardTitle>Écritures Comptables</CardTitle>
                  <CardDescription>Journal des opérations</CardDescription>
                </div>
                <Button 
                  className="bg-indigo-600 hover:bg-indigo-700"
                  onClick={() => setShowJournalDialog(true)}
                >
                  <Plus className="h-4 w-4 mr-2" />
                  Nouvelle Écriture
                </Button>
              </div>
            </CardHeader>
            <CardContent>
              {journalEntries.length === 0 ? (
                <div className="text-center py-8 text-gray-500">
                  Aucune écriture comptable. Cliquez sur "Nouvelle Écriture" pour commencer.
                </div>
              ) : (
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead>N° Écriture</TableHead>
                      <TableHead>Date</TableHead>
                      <TableHead>Journal</TableHead>
                      <TableHead>Description</TableHead>
                      <TableHead>Référence</TableHead>
                      <TableHead className="text-right">Débit</TableHead>
                      <TableHead className="text-right">Crédit</TableHead>
                      <TableHead>Statut</TableHead>
                      <TableHead>Actions</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {journalEntries.map((entry) => (
                      <TableRow key={entry.id}>
                        <TableCell className="font-medium">{entry.entry_number}</TableCell>
                        <TableCell>{new Date(entry.date).toLocaleDateString('fr-FR')}</TableCell>
                        <TableCell>{getJournalTypeBadge(entry.journal_type)}</TableCell>
                        <TableCell className="max-w-xs truncate">{entry.description}</TableCell>
                        <TableCell>{entry.reference || '-'}</TableCell>
                        <TableCell className="text-right font-medium">
                          {entry.total_debit.toLocaleString('fr-FR')} DA
                        </TableCell>
                        <TableCell className="text-right font-medium">
                          {entry.total_credit.toLocaleString('fr-FR')} DA
                        </TableCell>
                        <TableCell>{getStatusBadge(entry.status)}</TableCell>
                        <TableCell>
                          <Button
                            size="sm"
                            variant="outline"
                            onClick={() => {
                              setSelectedEntry(entry);
                              setShowJournalDetailsDialog(true);
                            }}
                          >
                            Détails
                          </Button>
                        </TableCell>
                      </TableRow>
                    ))}
                  </TableBody>
                </Table>
              )}
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>

      {/* Bank Dialog */}
      <Dialog open={showBankDialog} onOpenChange={setShowBankDialog}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Nouveau Compte Bancaire</DialogTitle>
            <DialogDescription>Ajouter un nouveau compte en banque</DialogDescription>
          </DialogHeader>
          <div className="grid gap-4 py-4">
            <div className="space-y-2">
              <Label>Nom de la Banque *</Label>
              <Input
                value={bankForm.bank_name}
                onChange={(e) => setBankForm({...bankForm, bank_name: e.target.value})}
                placeholder="Ex: BNA, CPA, BEA..."
              />
            </div>
            <div className="space-y-2">
              <Label>Numéro de Compte *</Label>
              <Input
                value={bankForm.account_number}
                onChange={(e) => setBankForm({...bankForm, account_number: e.target.value})}
                placeholder="Ex: 001234567890"
              />
            </div>
            <div className="space-y-2">
              <Label>RIB (20 chiffres) *</Label>
              <Input
                value={bankForm.rib}
                onChange={(e) => setBankForm({...bankForm, rib: e.target.value})}
                placeholder="Ex: 00712345678901234567"
                maxLength={20}
              />
            </div>
            <div className="space-y-2">
              <Label>IBAN (optionnel)</Label>
              <Input
                value={bankForm.iban}
                onChange={(e) => setBankForm({...bankForm, iban: e.target.value})}
                placeholder="Ex: DZ5900712345678901234567"
              />
            </div>
            <div className="space-y-2">
              <Label>Code SWIFT (optionnel)</Label>
              <Input
                value={bankForm.swift}
                onChange={(e) => setBankForm({...bankForm, swift: e.target.value})}
                placeholder="Ex: BNAADZAL"
              />
            </div>
            <div className="space-y-2">
              <Label>Solde Initial</Label>
              <Input
                type="number"
                value={bankForm.balance}
                onChange={(e) => setBankForm({...bankForm, balance: parseFloat(e.target.value) || 0})}
                placeholder="0"
              />
            </div>
          </div>
          <DialogFooter>
            <Button variant="outline" onClick={() => {setShowBankDialog(false); resetBankForm();}}>
              Annuler
            </Button>
            <Button 
              className="bg-indigo-600"
              onClick={handleCreateBank}
              disabled={!bankForm.bank_name || !bankForm.account_number || !bankForm.rib}
            >
              Créer
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Cash Dialog */}
      <Dialog open={showCashDialog} onOpenChange={setShowCashDialog}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Nouvelle Caisse</DialogTitle>
            <DialogDescription>Créer une nouvelle caisse</DialogDescription>
          </DialogHeader>
          <div className="grid gap-4 py-4">
            <div className="space-y-2">
              <Label>Nom de la Caisse *</Label>
              <Input
                value={cashForm.name}
                onChange={(e) => setCashForm({...cashForm, name: e.target.value})}
                placeholder="Ex: Caisse Principale, Caisse Atelier..."
              />
            </div>
            <div className="space-y-2">
              <Label>Responsable *</Label>
              <Input
                value={cashForm.responsible}
                onChange={(e) => setCashForm({...cashForm, responsible: e.target.value})}
                placeholder="Nom du responsable"
              />
            </div>
            <div className="space-y-2">
              <Label>Site (optionnel)</Label>
              <Input
                value={cashForm.site_id}
                onChange={(e) => setCashForm({...cashForm, site_id: e.target.value})}
                placeholder="Nom du site"
              />
            </div>
            <div className="space-y-2">
              <Label>Solde Initial</Label>
              <Input
                type="number"
                value={cashForm.balance}
                onChange={(e) => setCashForm({...cashForm, balance: parseFloat(e.target.value) || 0})}
                placeholder="0"
              />
            </div>
          </div>
          <DialogFooter>
            <Button variant="outline" onClick={() => {setShowCashDialog(false); resetCashForm();}}>
              Annuler
            </Button>
            <Button 
              className="bg-indigo-600"
              onClick={handleCreateCash}
              disabled={!cashForm.name || !cashForm.responsible}
            >
              Créer
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Journal Entry Dialog */}
      <Dialog open={showJournalDialog} onOpenChange={setShowJournalDialog}>
        <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle>Nouvelle Écriture Comptable</DialogTitle>
            <DialogDescription>
              Créer une nouvelle écriture. L'écriture doit être équilibrée (Débit = Crédit)
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div className="grid grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label>Type de Journal *</Label>
                <Select
                  value={journalForm.journal_type}
                  onValueChange={(value) => setJournalForm({...journalForm, journal_type: value})}
                >
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="sales">Ventes</SelectItem>
                    <SelectItem value="purchases">Achats</SelectItem>
                    <SelectItem value="bank">Banque</SelectItem>
                    <SelectItem value="cash">Caisse</SelectItem>
                    <SelectItem value="operations">Opérations Diverses</SelectItem>
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-2">
                <Label>Date *</Label>
                <Input
                  type="date"
                  value={journalForm.date}
                  onChange={(e) => setJournalForm({...journalForm, date: e.target.value})}
                />
              </div>
            </div>
            <div className="space-y-2">
              <Label>Description *</Label>
              <Input
                value={journalForm.description}
                onChange={(e) => setJournalForm({...journalForm, description: e.target.value})}
                placeholder="Ex: Facture client, Achat fournitures..."
              />
            </div>
            <div className="space-y-2">
              <Label>Référence</Label>
              <Input
                value={journalForm.reference}
                onChange={(e) => setJournalForm({...journalForm, reference: e.target.value})}
                placeholder="Ex: FAC-2025-001, BL-123..."
              />
            </div>

            <div className="space-y-3">
              <div className="flex items-center justify-between">
                <Label className="text-lg font-semibold">Lignes d'Écriture</Label>
                <Button
                  size="sm"
                  variant="outline"
                  onClick={handleAddJournalLine}
                >
                  <Plus className="h-4 w-4 mr-1" />
                  Ajouter ligne
                </Button>
              </div>

              <div className="border rounded-lg p-4 space-y-3 bg-gray-50">
                {journalForm.lines.map((line, index) => (
                  <div key={index} className="grid grid-cols-12 gap-2 items-end">
                    <div className="col-span-3 space-y-1">
                      <Label className="text-xs">Compte</Label>
                      <Input
                        placeholder="Ex: 411001"
                        value={line.account_code}
                        onChange={(e) => handleJournalLineChange(index, 'account_code', e.target.value)}
                      />
                    </div>
                    <div className="col-span-3 space-y-1">
                      <Label className="text-xs">Libellé Compte</Label>
                      <Input
                        placeholder="Ex: Client..."
                        value={line.account_name}
                        onChange={(e) => handleJournalLineChange(index, 'account_name', e.target.value)}
                      />
                    </div>
                    <div className="col-span-2 space-y-1">
                      <Label className="text-xs">Débit (DA)</Label>
                      <Input
                        type="number"
                        placeholder="0"
                        value={line.debit}
                        onChange={(e) => handleJournalLineChange(index, 'debit', parseFloat(e.target.value) || 0)}
                      />
                    </div>
                    <div className="col-span-2 space-y-1">
                      <Label className="text-xs">Crédit (DA)</Label>
                      <Input
                        type="number"
                        placeholder="0"
                        value={line.credit}
                        onChange={(e) => handleJournalLineChange(index, 'credit', parseFloat(e.target.value) || 0)}
                      />
                    </div>
                    <div className="col-span-1 space-y-1">
                      <Label className="text-xs invisible">Action</Label>
                      {journalForm.lines.length > 2 && (
                        <Button
                          size="sm"
                          variant="ghost"
                          className="text-red-600 h-9 w-9 p-0"
                          onClick={() => handleRemoveJournalLine(index)}
                        >
                          <Trash2 className="h-4 w-4" />
                        </Button>
                      )}
                    </div>
                    {line.description !== undefined && (
                      <div className="col-span-12 space-y-1">
                        <Input
                          placeholder="Description (optionnel)"
                          value={line.description}
                          onChange={(e) => handleJournalLineChange(index, 'description', e.target.value)}
                        />
                      </div>
                    )}
                  </div>
                ))}

                <div className="grid grid-cols-12 gap-2 pt-3 border-t border-gray-300">
                  <div className="col-span-6"></div>
                  <div className="col-span-2">
                    <div className="text-sm font-semibold">
                      Total Débit: {getTotalDebit().toLocaleString('fr-FR')} DA
                    </div>
                  </div>
                  <div className="col-span-2">
                    <div className="text-sm font-semibold">
                      Total Crédit: {getTotalCredit().toLocaleString('fr-FR')} DA
                    </div>
                  </div>
                  <div className="col-span-2">
                    <div className={`text-sm font-semibold ${Math.abs(getTotalDebit() - getTotalCredit()) < 0.01 ? 'text-green-600' : 'text-red-600'}`}>
                      {Math.abs(getTotalDebit() - getTotalCredit()) < 0.01 ? '✓ Équilibré' : '✗ Non équilibré'}
                    </div>
                  </div>
                </div>
              </div>
            </div>
          </div>
          <DialogFooter>
            <Button variant="outline" onClick={() => {setShowJournalDialog(false); resetJournalForm();}}>
              Annuler
            </Button>
            <Button
              className="bg-indigo-600 hover:bg-indigo-700"
              onClick={handleCreateJournalEntry}
              disabled={!journalForm.description || Math.abs(getTotalDebit() - getTotalCredit()) > 0.01}
            >
              Créer Écriture
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Journal Entry Details Dialog */}
      <Dialog open={showJournalDetailsDialog} onOpenChange={setShowJournalDetailsDialog}>
        <DialogContent className="max-w-3xl">
          <DialogHeader>
            <DialogTitle>Détails de l'Écriture</DialogTitle>
          </DialogHeader>
          {selectedEntry && (
            <div className="space-y-4">
              <div className="grid grid-cols-2 gap-4 bg-gray-50 p-4 rounded-lg">
                <div>
                  <p className="text-sm text-gray-500">N° Écriture</p>
                  <p className="font-semibold">{selectedEntry.entry_number}</p>
                </div>
                <div>
                  <p className="text-sm text-gray-500">Date</p>
                  <p className="font-semibold">{new Date(selectedEntry.date).toLocaleDateString('fr-FR')}</p>
                </div>
                <div>
                  <p className="text-sm text-gray-500">Type de Journal</p>
                  <div className="mt-1">{getJournalTypeBadge(selectedEntry.journal_type)}</div>
                </div>
                <div>
                  <p className="text-sm text-gray-500">Statut</p>
                  <div className="mt-1">{getStatusBadge(selectedEntry.status)}</div>
                </div>
                <div className="col-span-2">
                  <p className="text-sm text-gray-500">Description</p>
                  <p className="font-medium">{selectedEntry.description}</p>
                </div>
                {selectedEntry.reference && (
                  <div className="col-span-2">
                    <p className="text-sm text-gray-500">Référence</p>
                    <p className="font-medium">{selectedEntry.reference}</p>
                  </div>
                )}
              </div>

              <div>
                <h4 className="font-semibold mb-3">Lignes d'Écriture</h4>
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead>Compte</TableHead>
                      <TableHead>Libellé</TableHead>
                      <TableHead className="text-right">Débit</TableHead>
                      <TableHead className="text-right">Crédit</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {selectedEntry.lines.map((line, index) => (
                      <TableRow key={index}>
                        <TableCell className="font-mono">{line.account_code}</TableCell>
                        <TableCell>{line.account_name}</TableCell>
                        <TableCell className="text-right font-medium">
                          {line.debit > 0 ? `${line.debit.toLocaleString('fr-FR')} DA` : '-'}
                        </TableCell>
                        <TableCell className="text-right font-medium">
                          {line.credit > 0 ? `${line.credit.toLocaleString('fr-FR')} DA` : '-'}
                        </TableCell>
                      </TableRow>
                    ))}
                    <TableRow className="bg-gray-100 font-semibold">
                      <TableCell colSpan={2}>Total</TableCell>
                      <TableCell className="text-right">{selectedEntry.total_debit.toLocaleString('fr-FR')} DA</TableCell>
                      <TableCell className="text-right">{selectedEntry.total_credit.toLocaleString('fr-FR')} DA</TableCell>
                    </TableRow>
                  </TableBody>
                </Table>
              </div>
            </div>
          )}
          <DialogFooter>
            <Button variant="outline" onClick={() => {
              setShowJournalDetailsDialog(false);
              setSelectedEntry(null);
            }}>
              Fermer
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
};

export default FinancePage;
