import React, { useState } from 'react';
import api from '@/services/api';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Badge } from '@/components/ui/badge';
import { Sparkles, TrendingUp, AlertTriangle, Users, Wrench, DollarSign, Brain, Zap } from 'lucide-react';

const AIInsightsPage = () => {
  const [loading, setLoading] = useState({});
  const [generalInsights, setGeneralInsights] = useState(null);
  const [workshopAnomalies, setWorkshopAnomalies] = useState(null);
  const [financialAnalysis, setFinancialAnalysis] = useState(null);
  const [cashflowPrediction, setCashflowPrediction] = useState(null);
  const [hrAnalysis, setHrAnalysis] = useState(null);

  const handleGetGeneralInsights = async () => {
    setLoading({ ...loading, general: true });
    try {
      const response = await api.get('/ai/insights/general');
      setGeneralInsights(response.data);
    } catch (error) {
      console.error('Error:', error);
      alert(error.response?.data?.detail || 'Erreur lors de la récupération des insights');
    } finally {
      setLoading({ ...loading, general: false });
    }
  };

  const handleDetectWorkshopAnomalies = async () => {
    setLoading({ ...loading, workshop: true });
    try {
      const response = await api.post('/ai/workshop/detect-anomalies');
      setWorkshopAnomalies(response.data);
    } catch (error) {
      console.error('Error:', error);
      alert(error.response?.data?.detail || 'Erreur lors de la détection');
    } finally {
      setLoading({ ...loading, workshop: false });
    }
  };

  const handleAnalyzeFinances = async () => {
    setLoading({ ...loading, finance: true });
    try {
      const response = await api.post('/ai/finance/analyze-transactions', null, {
        params: { period_days: 30 }
      });
      setFinancialAnalysis(response.data);
    } catch (error) {
      console.error('Error:', error);
      alert(error.response?.data?.detail || 'Erreur lors de l\'analyse');
    } finally {
      setLoading({ ...loading, finance: false });
    }
  };

  const handlePredictCashflow = async () => {
    setLoading({ ...loading, cashflow: true });
    try {
      const response = await api.post('/ai/finance/predict-cashflow', null, {
        params: { months_ahead: 3 }
      });
      setCashflowPrediction(response.data);
    } catch (error) {
      console.error('Error:', error);
      alert(error.response?.data?.detail || 'Erreur lors de la prédiction');
    } finally {
      setLoading({ ...loading, cashflow: false });
    }
  };

  const handleAnalyzeAbsences = async () => {
    setLoading({ ...loading, hr: true });
    try {
      const response = await api.post('/ai/hr/analyze-absences');
      setHrAnalysis(response.data);
    } catch (error) {
      console.error('Error:', error);
      alert(error.response?.data?.detail || 'Erreur lors de l\'analyse');
    } finally {
      setLoading({ ...loading, hr: false });
    }
  };

  const getSeverityBadge = (severity) => {
    const badges = {
      low: { class: 'bg-yellow-100 text-yellow-800', label: 'Faible' },
      medium: { class: 'bg-orange-100 text-orange-800', label: 'Moyen' },
      high: { class: 'bg-red-100 text-red-800', label: 'Élevé' }
    };
    const badge = badges[severity] || badges.low;
    return <Badge className={badge.class}>{badge.label}</Badge>;
  };

  const getHealthStatusBadge = (status) => {
    const badges = {
      excellent: { class: 'bg-green-100 text-green-800', label: 'Excellent' },
      good: { class: 'bg-blue-100 text-blue-800', label: 'Bon' },
      moderate: { class: 'bg-yellow-100 text-yellow-800', label: 'Modéré' },
      poor: { class: 'bg-red-100 text-red-800', label: 'Faible' }
    };
    const badge = badges[status] || badges.moderate;
    return <Badge className={badge.class}>{badge.label}</Badge>;
  };

  return (
    <div className="container mx-auto py-8 px-4">
      <div className="mb-8">
        <div className="flex items-center gap-3 mb-2">
          <Brain className="h-8 w-8 text-indigo-600" />
          <h1 className="text-3xl font-bold text-gray-900">Intelligence Artificielle</h1>
        </div>
        <p className="text-gray-600">Analyse prédictive et détection d'anomalies propulsée par IA</p>
      </div>

      <Tabs defaultValue="general" className="space-y-6">
        <TabsList className="grid w-full grid-cols-5">
          <TabsTrigger value="general">
            <Sparkles className="h-4 w-4 mr-2" />
            Vue Générale
          </TabsTrigger>
          <TabsTrigger value="workshop">
            <Wrench className="h-4 w-4 mr-2" />
            Atelier
          </TabsTrigger>
          <TabsTrigger value="finance">
            <DollarSign className="h-4 w-4 mr-2" />
            Finance
          </TabsTrigger>
          <TabsTrigger value="cashflow">
            <TrendingUp className="h-4 w-4 mr-2" />
            Trésorerie
          </TabsTrigger>
          <TabsTrigger value="hr">
            <Users className="h-4 w-4 mr-2" />
            RH
          </TabsTrigger>
        </TabsList>

        {/* GENERAL INSIGHTS */}
        <TabsContent value="general">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Sparkles className="h-5 w-5 text-indigo-600" />
                Insights Généraux de l'Entreprise
              </CardTitle>
              <CardDescription>Vue d'ensemble intelligente de la santé de votre entreprise</CardDescription>
            </CardHeader>
            <CardContent>
              <Button
                onClick={handleGetGeneralInsights}
                disabled={loading.general}
                className="bg-indigo-600 hover:bg-indigo-700 mb-6"
              >
                <Zap className="h-4 w-4 mr-2" />
                {loading.general ? 'Analyse en cours...' : 'Générer Insights IA'}
              </Button>

              {generalInsights && (
                <div className="space-y-6">
                  {/* Health Score */}
                  {generalInsights.insights?.health_score && (
                    <div className="bg-gradient-to-r from-indigo-50 to-purple-50 p-6 rounded-lg border border-indigo-200">
                      <div className="flex items-center justify-between mb-4">
                        <h3 className="text-lg font-semibold">Score de Santé</h3>
                        {getHealthStatusBadge(generalInsights.insights.health_status)}
                      </div>
                      <div className="flex items-center gap-4">
                        <div className="text-5xl font-bold text-indigo-600">
                          {generalInsights.insights.health_score}
                          <span className="text-2xl text-gray-500">/100</span>
                        </div>
                        <div className="flex-1">
                          <div className="w-full bg-gray-200 rounded-full h-4">
                            <div
                              className="bg-indigo-600 h-4 rounded-full transition-all"
                              style={{ width: `${generalInsights.insights.health_score}%` }}
                            />
                          </div>
                        </div>
                      </div>
                    </div>
                  )}

                  {/* Executive Summary */}
                  {generalInsights.insights?.executive_summary && (
                    <Card className="border-blue-200 bg-blue-50">
                      <CardHeader>
                        <CardTitle className="text-base text-blue-900">Résumé Exécutif</CardTitle>
                      </CardHeader>
                      <CardContent>
                        <p className="text-blue-800">{generalInsights.insights.executive_summary}</p>
                      </CardContent>
                    </Card>
                  )}

                  {/* Alerts */}
                  {generalInsights.insights?.alerts && generalInsights.insights.alerts.length > 0 && (
                    <Card>
                      <CardHeader>
                        <CardTitle className="text-base flex items-center gap-2">
                          <AlertTriangle className="h-5 w-5 text-orange-600" />
                          Alertes & Actions Prioritaires
                        </CardTitle>
                      </CardHeader>
                      <CardContent>
                        <div className="space-y-3">
                          {generalInsights.insights.alerts.map((alert, idx) => (
                            <div key={idx} className="flex items-start gap-3 p-3 bg-gray-50 rounded-lg">
                              <Badge className={
                                alert.priority === 'high' ? 'bg-red-100 text-red-800' :
                                alert.priority === 'medium' ? 'bg-orange-100 text-orange-800' :
                                'bg-yellow-100 text-yellow-800'
                              }>
                                {alert.priority === 'high' ? 'Urgent' : alert.priority === 'medium' ? 'Important' : 'Info'}
                              </Badge>
                              <div className="flex-1">
                                <p className="font-medium">{alert.message}</p>
                                {alert.action && <p className="text-sm text-gray-600 mt-1">Action: {alert.action}</p>}
                              </div>
                            </div>
                          ))}
                        </div>
                      </CardContent>
                    </Card>
                  )}

                  {/* Recommendations */}
                  {generalInsights.insights?.recommendations && generalInsights.insights.recommendations.length > 0 && (
                    <Card>
                      <CardHeader>
                        <CardTitle className="text-base">Recommandations Stratégiques</CardTitle>
                      </CardHeader>
                      <CardContent>
                        <ul className="space-y-2">
                          {generalInsights.insights.recommendations.map((rec, idx) => (
                            <li key={idx} className="flex items-start gap-2">
                              <span className="text-indigo-600 font-bold">→</span>
                              <span>{rec}</span>
                            </li>
                          ))}
                        </ul>
                      </CardContent>
                    </Card>
                  )}

                  {/* Opportunities */}
                  {generalInsights.insights?.opportunities && generalInsights.insights.opportunities.length > 0 && (
                    <Card className="border-green-200 bg-green-50">
                      <CardHeader>
                        <CardTitle className="text-base text-green-900">Opportunités Identifiées</CardTitle>
                      </CardHeader>
                      <CardContent>
                        <ul className="space-y-2 text-green-800">
                          {generalInsights.insights.opportunities.map((opp, idx) => (
                            <li key={idx} className="flex items-start gap-2">
                              <span className="text-green-600">✓</span>
                              <span>{opp}</span>
                            </li>
                          ))}
                        </ul>
                      </CardContent>
                    </Card>
                  )}
                </div>
              )}
            </CardContent>
          </Card>
        </TabsContent>

        {/* WORKSHOP ANOMALIES */}
        <TabsContent value="workshop">
          <Card>
            <CardHeader>
              <CardTitle>Détection d'Anomalies - Atelier</CardTitle>
              <CardDescription>Analyse IA des ordres de réparation</CardDescription>
            </CardHeader>
            <CardContent>
              <Button
                onClick={handleDetectWorkshopAnomalies}
                disabled={loading.workshop}
                className="bg-indigo-600 hover:bg-indigo-700 mb-6"
              >
                {loading.workshop ? 'Analyse en cours...' : 'Détecter Anomalies'}
              </Button>

              {workshopAnomalies && (
                <div className="space-y-4">
                  {workshopAnomalies.summary && (
                    <div className="bg-blue-50 p-4 rounded-lg">
                      <h4 className="font-semibold mb-2">Résumé:</h4>
                      <p>{workshopAnomalies.summary}</p>
                    </div>
                  )}

                  {workshopAnomalies.anomalies && workshopAnomalies.anomalies.length > 0 ? (
                    <div className="space-y-3">
                      {workshopAnomalies.anomalies.map((anomaly, idx) => (
                        <Card key={idx} className="border-l-4 border-orange-400">
                          <CardContent className="pt-4">
                            <div className="flex items-start justify-between mb-2">
                              <span className="font-medium">{anomaly.repair_number}</span>
                              {getSeverityBadge(anomaly.severity)}
                            </div>
                            <p className="text-sm text-gray-700 mb-2">{anomaly.description}</p>
                            {anomaly.recommendation && (
                              <p className="text-sm text-indigo-600">
                                <strong>Recommandation:</strong> {anomaly.recommendation}
                              </p>
                            )}
                          </CardContent>
                        </Card>
                      ))}
                    </div>
                  ) : (
                    <p className="text-gray-500 text-center py-4">Aucune anomalie détectée</p>
                  )}
                </div>
              )}
            </CardContent>
          </Card>
        </TabsContent>

        {/* FINANCIAL ANALYSIS */}
        <TabsContent value="finance">
          <Card>
            <CardHeader>
              <CardTitle>Analyse Financière IA</CardTitle>
              <CardDescription>Détection d'anomalies et patterns dans les transactions</CardDescription>
            </CardHeader>
            <CardContent>
              <Button
                onClick={handleAnalyzeFinances}
                disabled={loading.finance}
                className="bg-indigo-600 hover:bg-indigo-700 mb-6"
              >
                {loading.finance ? 'Analyse en cours...' : 'Analyser (30 derniers jours)'}
              </Button>

              {financialAnalysis && financialAnalysis.analysis && (
                <div className="space-y-4">
                  {financialAnalysis.analysis.risk_score !== undefined && (
                    <div className="bg-gradient-to-r from-red-50 to-orange-50 p-4 rounded-lg">
                      <div className="flex items-center justify-between">
                        <span className="font-semibold">Score de Risque:</span>
                        <span className="text-2xl font-bold">{financialAnalysis.analysis.risk_score}/100</span>
                      </div>
                    </div>
                  )}

                  {financialAnalysis.analysis.summary && (
                    <Card className="bg-blue-50 border-blue-200">
                      <CardContent className="pt-4">
                        <p>{financialAnalysis.analysis.summary}</p>
                      </CardContent>
                    </Card>
                  )}

                  {financialAnalysis.analysis.anomalies && financialAnalysis.analysis.anomalies.length > 0 && (
                    <div>
                      <h4 className="font-semibold mb-3">Anomalies Détectées:</h4>
                      <div className="space-y-2">
                        {financialAnalysis.analysis.anomalies.map((anomaly, idx) => (
                          <div key={idx} className="flex items-start gap-2 p-3 bg-gray-50 rounded">
                            {getSeverityBadge(anomaly.severity)}
                            <div>
                              <p className="font-medium">{anomaly.type}</p>
                              <p className="text-sm text-gray-600">{anomaly.description}</p>
                            </div>
                          </div>
                        ))}
                      </div>
                    </div>
                  )}

                  {financialAnalysis.analysis.recommendations && financialAnalysis.analysis.recommendations.length > 0 && (
                    <Card>
                      <CardHeader>
                        <CardTitle className="text-base">Recommandations</CardTitle>
                      </CardHeader>
                      <CardContent>
                        <ul className="space-y-1">
                          {financialAnalysis.analysis.recommendations.map((rec, idx) => (
                            <li key={idx} className="flex items-start gap-2">
                              <span className="text-indigo-600">→</span>
                              <span className="text-sm">{rec}</span>
                            </li>
                          ))}
                        </ul>
                      </CardContent>
                    </Card>
                  )}
                </div>
              )}
            </CardContent>
          </Card>
        </TabsContent>

        {/* CASHFLOW PREDICTION */}
        <TabsContent value="cashflow">
          <Card>
            <CardHeader>
              <CardTitle>Prédiction de Trésorerie</CardTitle>
              <CardDescription>Prévisions IA basées sur l'historique</CardDescription>
            </CardHeader>
            <CardContent>
              <Button
                onClick={handlePredictCashflow}
                disabled={loading.cashflow}
                className="bg-indigo-600 hover:bg-indigo-700 mb-6"
              >
                {loading.cashflow ? 'Analyse en cours...' : 'Prédire (3 prochains mois)'}
              </Button>

              {cashflowPrediction && cashflowPrediction.prediction && (
                <div className="space-y-4">
                  {cashflowPrediction.prediction.summary && (
                    <Card className="bg-blue-50 border-blue-200">
                      <CardContent className="pt-4">
                        <p>{cashflowPrediction.prediction.summary}</p>
                      </CardContent>
                    </Card>
                  )}

                  {cashflowPrediction.prediction.predictions && cashflowPrediction.prediction.predictions.length > 0 && (
                    <div>
                      <h4 className="font-semibold mb-3">Prédictions Mensuelles:</h4>
                      <div className="space-y-3">
                        {cashflowPrediction.prediction.predictions.map((pred, idx) => (
                          <Card key={idx}>
                            <CardContent className="pt-4">
                              <div className="flex items-center justify-between mb-2">
                                <span className="font-semibold">{pred.month}</span>
                                <Badge className={pred.confidence === 'high' ? 'bg-green-100 text-green-800' : 'bg-yellow-100 text-yellow-800'}>
                                  Confiance: {pred.confidence === 'high' ? 'Élevée' : pred.confidence === 'medium' ? 'Moyenne' : 'Faible'}
                                </Badge>
                              </div>
                              <div className="grid grid-cols-3 gap-4 text-sm">
                                <div>
                                  <p className="text-gray-500">Entrées</p>
                                  <p className="font-semibold text-green-600">+{pred.predicted_inflow?.toLocaleString()} DA</p>
                                </div>
                                <div>
                                  <p className="text-gray-500">Sorties</p>
                                  <p className="font-semibold text-red-600">-{pred.predicted_outflow?.toLocaleString()} DA</p>
                                </div>
                                <div>
                                  <p className="text-gray-500">Net</p>
                                  <p className={`font-semibold ${pred.net_cashflow >= 0 ? 'text-green-600' : 'text-red-600'}`}>
                                    {pred.net_cashflow?.toLocaleString()} DA
                                  </p>
                                </div>
                              </div>
                            </CardContent>
                          </Card>
                        ))}
                      </div>
                    </div>
                  )}

                  {cashflowPrediction.prediction.trends && cashflowPrediction.prediction.trends.length > 0 && (
                    <Card>
                      <CardHeader>
                        <CardTitle className="text-base">Tendances</CardTitle>
                      </CardHeader>
                      <CardContent>
                        <ul className="space-y-1">
                          {cashflowPrediction.prediction.trends.map((trend, idx) => (
                            <li key={idx} className="text-sm">• {trend}</li>
                          ))}
                        </ul>
                      </CardContent>
                    </Card>
                  )}

                  {cashflowPrediction.prediction.risks && cashflowPrediction.prediction.risks.length > 0 && (
                    <Card className="border-red-200 bg-red-50">
                      <CardHeader>
                        <CardTitle className="text-base text-red-900">Risques Identifiés</CardTitle>
                      </CardHeader>
                      <CardContent>
                        <ul className="space-y-1 text-red-800">
                          {cashflowPrediction.prediction.risks.map((risk, idx) => (
                            <li key={idx} className="text-sm">⚠ {risk}</li>
                          ))}
                        </ul>
                      </CardContent>
                    </Card>
                  )}
                </div>
              )}
            </CardContent>
          </Card>
        </TabsContent>

        {/* HR ANALYSIS */}
        <TabsContent value="hr">
          <Card>
            <CardHeader>
              <CardTitle>Analyse RH - Absences & Congés</CardTitle>
              <CardDescription>Détection de patterns et prédictions</CardDescription>
            </CardHeader>
            <CardContent>
              <Button
                onClick={handleAnalyzeAbsences}
                disabled={loading.hr}
                className="bg-indigo-600 hover:bg-indigo-700 mb-6"
              >
                {loading.hr ? 'Analyse en cours...' : 'Analyser Absences'}
              </Button>

              {hrAnalysis && hrAnalysis.analysis && (
                <div className="space-y-4">
                  {hrAnalysis.analysis.summary && (
                    <Card className="bg-blue-50 border-blue-200">
                      <CardContent className="pt-4">
                        <p>{hrAnalysis.analysis.summary}</p>
                      </CardContent>
                    </Card>
                  )}

                  {hrAnalysis.analysis.patterns && hrAnalysis.analysis.patterns.length > 0 && (
                    <Card>
                      <CardHeader>
                        <CardTitle className="text-base">Patterns Détectés</CardTitle>
                      </CardHeader>
                      <CardContent>
                        <ul className="space-y-1">
                          {hrAnalysis.analysis.patterns.map((pattern, idx) => (
                            <li key={idx} className="text-sm">• {pattern}</li>
                          ))}
                        </ul>
                      </CardContent>
                    </Card>
                  )}

                  {hrAnalysis.analysis.anomalies && hrAnalysis.analysis.anomalies.length > 0 && (
                    <div>
                      <h4 className="font-semibold mb-3">Anomalies:</h4>
                      <div className="space-y-2">
                        {hrAnalysis.analysis.anomalies.map((anomaly, idx) => (
                          <div key={idx} className="flex items-start gap-2 p-3 bg-gray-50 rounded">
                            {getSeverityBadge(anomaly.severity)}
                            <div>
                              <p className="font-medium">{anomaly.employee}</p>
                              <p className="text-sm text-gray-600">{anomaly.issue}</p>
                            </div>
                          </div>
                        ))}
                      </div>
                    </div>
                  )}

                  {hrAnalysis.analysis.recommendations && hrAnalysis.analysis.recommendations.length > 0 && (
                    <Card>
                      <CardHeader>
                        <CardTitle className="text-base">Recommandations</CardTitle>
                      </CardHeader>
                      <CardContent>
                        <ul className="space-y-1">
                          {hrAnalysis.analysis.recommendations.map((rec, idx) => (
                            <li key={idx} className="flex items-start gap-2">
                              <span className="text-indigo-600">→</span>
                              <span className="text-sm">{rec}</span>
                            </li>
                          ))}
                        </ul>
                      </CardContent>
                    </Card>
                  )}
                </div>
              )}
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>
    </div>
  );
};

export default AIInsightsPage;
