import React from "react";
import "@/App.css";
import { BrowserRouter, Routes, Route, Navigate } from "react-router-dom";
import { AuthProvider, useAuth } from "@/contexts/AuthContext";
import { ThemeProvider } from "@/contexts/ThemeContext";
import "./i18n";

// Layouts
import AdminLayout from "@/layouts/AdminLayout";
import ClientLayout from "@/layouts/ClientLayout";

// Public Pages
import HomePage from "@/pages/public/HomePage";

// Auth Pages
import Login from "@/pages/auth/Login";

// Admin Pages
import AdminDashboard from "@/pages/admin/Dashboard";
import ProductsPage from "@/pages/admin/ProductsPage";
import OrdersPage from "@/pages/admin/OrdersPage";
import ClientsPage from "@/pages/admin/ClientsPage";
import CategoriesPage from "@/pages/admin/CategoriesPage";
import BrandsPage from "@/pages/admin/BrandsPage";
import UsersPage from "@/pages/admin/UsersPage";
import ReportsPage from "@/pages/admin/ReportsPage";
import StockManagement from "@/pages/admin/StockManagement";
import ImportsManagement from "@/pages/admin/ImportsManagement";
import WorkshopManagement from "@/pages/admin/WorkshopManagement";
import FinancePage from "@/pages/admin/FinancePage";
import FiscalPage from "@/pages/admin/FiscalPage";
import HRPage from "@/pages/admin/HRPage";
import GEDPage from "@/pages/admin/GEDPage";
import AIInsightsPage from "@/pages/admin/AIInsightsPage";

// Client Pages
import ClientDashboard from "@/pages/client/ClientDashboard";
import ProductCatalog from "@/pages/client/ProductCatalog";
import Cart from "@/pages/client/Cart";
import ClientOrders from "@/pages/client/ClientOrders";

// Protected Route Component
const ProtectedRoute = ({ children, allowedRoles }) => {
  const { user, loading } = useAuth();

  if (loading) {
    return (
      <div className="flex items-center justify-center h-screen">
        <div className="text-lg">Chargement...</div>
      </div>
    );
  }

  if (!user) {
    return <Navigate to="/login" replace />;
  }

  if (allowedRoles && !allowedRoles.includes(user.role)) {
    return <Navigate to="/login" replace />;
  }

  return children;
};

function App() {
  return (
    <BrowserRouter>
      <ThemeProvider>
        <AuthProvider>
          <Routes>
            {/* Public Routes */}
            <Route path="/" element={<HomePage />} />
            <Route path="/login" element={<Login />} />

          {/* Admin Routes */}
          <Route
            path="/admin"
            element={
              <ProtectedRoute allowedRoles={['superadmin', 'admin', 'manager']}>
                <AdminLayout />
              </ProtectedRoute>
            }
          >
            <Route index element={<Navigate to="/admin/dashboard" replace />} />
            <Route path="dashboard" element={<AdminDashboard />} />
            <Route path="products" element={<ProductsPage />} />
            <Route path="stock" element={<StockManagement />} />
            <Route path="imports" element={<ImportsManagement />} />
            <Route path="workshop" element={<WorkshopManagement />} />
            <Route path="orders" element={<OrdersPage />} />
            <Route path="clients" element={<ClientsPage />} />
            <Route path="finance" element={<FinancePage />} />
            <Route path="fiscal" element={<FiscalPage />} />
            <Route path="hr" element={<HRPage />} />
            <Route path="documents" element={<GEDPage />} />
            <Route path="ai-insights" element={<AIInsightsPage />} />
            <Route path="reports" element={<ReportsPage />} />
            <Route path="categories" element={<CategoriesPage />} />
            <Route path="brands" element={<BrandsPage />} />
            <Route path="users" element={<UsersPage />} />
          </Route>

          {/* Client Routes */}
          <Route
            path="/client"
            element={
              <ProtectedRoute allowedRoles={['client']}>
                <ClientLayout />
              </ProtectedRoute>
            }
          >
            <Route index element={<Navigate to="/client/dashboard" replace />} />
            <Route path="dashboard" element={<ClientDashboard />} />
            <Route path="catalog" element={<ProductCatalog />} />
            <Route path="cart" element={<Cart />} />
            <Route path="orders" element={<ClientOrders />} />
          </Route>

          {/* Catch all */}
          <Route path="*" element={<Navigate to="/" replace />} />
        </Routes>
      </AuthProvider>
    </ThemeProvider>
  </BrowserRouter>
  );
}

export default App;
