from fastapi import APIRouter, HTTPException, Depends
from typing import List, Optional
from datetime import datetime, timezone
from models import User, UserCreate, UserResponse, UserRole, UserPermissions
from dependencies import get_current_user, get_db
from permissions import get_default_permissions, get_user_permissions, PERMISSION_TEMPLATES
from motor.motor_asyncio import AsyncIOMotorDatabase
import bcrypt

router = APIRouter(prefix="/users", tags=["users"])

# ==================== USERS MANAGEMENT (SUPERADMIN ONLY) ====================

@router.get("/", response_model=List[UserResponse])
async def get_users(
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    """Liste tous les utilisateurs (accessible au superadmin et pour consultation aux autres admins)"""
    if current_user.get("role") not in ["superadmin", "admin", "manager"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    users = await db.users.find({}, {"_id": 0, "password_hash": 0}).to_list(None)
    return users

@router.get("/{user_id}", response_model=UserResponse)
async def get_user(
    user_id: str,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    """Récupère un utilisateur par ID"""
    if current_user.get("role") not in ["superadmin", "admin"]:
        # Un utilisateur peut consulter son propre profil
        if current_user.get("id") != user_id:
            raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    user = await db.users.find_one({"id": user_id}, {"_id": 0, "password_hash": 0})
    if not user:
        raise HTTPException(status_code=404, detail="User not found")
    
    return user

@router.put("/{user_id}")
async def update_user(
    user_id: str,
    user_data: dict,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    """Met à jour un utilisateur (SUPERADMIN uniquement pour les permissions)"""
    # Seul le SUPERADMIN peut modifier les permissions
    if "permissions" in user_data and current_user.get("role") != "superadmin":
        raise HTTPException(status_code=403, detail="Only superadmin can modify permissions")
    
    # Les admins peuvent modifier les autres champs (sauf role et permissions)
    if current_user.get("role") not in ["superadmin", "admin"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    # Ne pas permettre la modification du mot de passe via cette route
    if "password" in user_data:
        del user_data["password"]
    if "password_hash" in user_data:
        del user_data["password_hash"]
    
    user_data["updated_at"] = datetime.now(timezone.utc).isoformat()
    
    result = await db.users.update_one(
        {"id": user_id},
        {"$set": user_data}
    )
    
    if result.modified_count == 0:
        raise HTTPException(status_code=404, detail="User not found or no changes made")
    
    return {"message": "User updated successfully"}

@router.patch("/{user_id}/permissions")
async def update_user_permissions(
    user_id: str,
    permissions: UserPermissions,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    """Met à jour les permissions d'un utilisateur (SUPERADMIN uniquement)"""
    if current_user.get("role") != "superadmin":
        raise HTTPException(status_code=403, detail="Only superadmin can modify permissions")
    
    permissions_dict = permissions.model_dump()
    
    result = await db.users.update_one(
        {"id": user_id},
        {
            "$set": {
                "permissions": permissions_dict,
                "updated_at": datetime.now(timezone.utc).isoformat()
            }
        }
    )
    
    if result.modified_count == 0:
        raise HTTPException(status_code=404, detail="User not found")
    
    return {"message": "Permissions updated successfully"}

@router.patch("/{user_id}/status")
async def toggle_user_status(
    user_id: str,
    status_data: dict,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    """Active/Désactive un utilisateur (SUPERADMIN uniquement)"""
    if current_user.get("role") != "superadmin":
        raise HTTPException(status_code=403, detail="Only superadmin can change user status")
    
    # Ne pas permettre la désactivation de son propre compte
    if user_id == current_user.get("id"):
        raise HTTPException(status_code=400, detail="Cannot deactivate your own account")
    
    is_active = status_data.get("is_active", True)
    
    result = await db.users.update_one(
        {"id": user_id},
        {
            "$set": {
                "is_active": is_active,
                "updated_at": datetime.now(timezone.utc).isoformat()
            }
        }
    )
    
    if result.modified_count == 0:
        raise HTTPException(status_code=404, detail="User not found")
    
    return {"message": f"User {'activated' if is_active else 'deactivated'} successfully"}

@router.delete("/{user_id}")
async def delete_user(
    user_id: str,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    """Supprime un utilisateur (SUPERADMIN uniquement)"""
    if current_user.get("role") != "superadmin":
        raise HTTPException(status_code=403, detail="Only superadmin can delete users")
    
    # Ne pas permettre la suppression de son propre compte
    if user_id == current_user.get("id"):
        raise HTTPException(status_code=400, detail="Cannot delete your own account")
    
    result = await db.users.delete_one({"id": user_id})
    
    if result.deleted_count == 0:
        raise HTTPException(status_code=404, detail="User not found")
    
    return {"message": "User deleted successfully"}

# ==================== PERMISSION TEMPLATES ====================

@router.get("/templates/permissions")
async def get_permission_templates(
    current_user: dict = Depends(get_current_user)
):
    """Récupère les templates de permissions prédéfinis (SUPERADMIN uniquement)"""
    if current_user.get("role") != "superadmin":
        raise HTTPException(status_code=403, detail="Only superadmin can access permission templates")
    
    return PERMISSION_TEMPLATES

# ==================== DEFAULT PERMISSIONS BY ROLE ====================

@router.get("/permissions/defaults/{role}")
async def get_role_default_permissions(
    role: str,
    current_user: dict = Depends(get_current_user)
):
    """Récupère les permissions par défaut pour un rôle (SUPERADMIN uniquement)"""
    if current_user.get("role") != "superadmin":
        raise HTTPException(status_code=403, detail="Only superadmin can access default permissions")
    
    try:
        user_role = UserRole(role)
        default_perms = get_default_permissions(user_role)
        return default_perms.model_dump()
    except ValueError:
        raise HTTPException(status_code=400, detail=f"Invalid role: {role}")
