from fastapi import APIRouter, HTTPException, Depends
from typing import List, Optional
from datetime import datetime, timezone
from models import (
    Employee, EmployeeCreate, EmployeeStatus, ContractType,
    Attendance, LeaveRequest, LeaveType, AbsenceStatus,
    Payslip, Training
)
from dependencies import get_current_user, get_db
from motor.motor_asyncio import AsyncIOMotorDatabase

router = APIRouter(prefix="/hr", tags=["hr"])

# ==================== EMPLOYÉS ====================
@router.get("/employees", response_model=List[Employee])
async def get_employees(
    status: Optional[str] = None,
    department: Optional[str] = None,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    if current_user.get("role") not in ["superadmin", "admin", "manager"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    query = {}
    if status:
        query["status"] = status
    if department:
        query["department"] = department
    
    employees = await db.employees.find(query, {"_id": 0}).sort("last_name", 1).to_list(None)
    
    for emp in employees:
        if isinstance(emp.get('date_of_birth'), str):
            emp['date_of_birth'] = datetime.fromisoformat(emp['date_of_birth'])
        if isinstance(emp.get('hire_date'), str):
            emp['hire_date'] = datetime.fromisoformat(emp['hire_date'])
        if emp.get('contract_end_date') and isinstance(emp['contract_end_date'], str):
            emp['contract_end_date'] = datetime.fromisoformat(emp['contract_end_date'])
    
    return employees

@router.get("/employees/{employee_id}", response_model=Employee)
async def get_employee(
    employee_id: str,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    if current_user.get("role") not in ["superadmin", "admin", "manager"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    employee = await db.employees.find_one({"id": employee_id}, {"_id": 0})
    if not employee:
        raise HTTPException(status_code=404, detail="Employee not found")
    
    if isinstance(employee.get('date_of_birth'), str):
        employee['date_of_birth'] = datetime.fromisoformat(employee['date_of_birth'])
    if isinstance(employee.get('hire_date'), str):
        employee['hire_date'] = datetime.fromisoformat(employee['hire_date'])
    if employee.get('contract_end_date') and isinstance(employee['contract_end_date'], str):
        employee['contract_end_date'] = datetime.fromisoformat(employee['contract_end_date'])
    
    return employee

@router.post("/employees", response_model=Employee)
async def create_employee(
    employee_data: EmployeeCreate,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    if current_user.get("role") not in ["superadmin", "admin"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    # Check if employee number already exists
    existing = await db.employees.find_one({"employee_number": employee_data.employee_number})
    if existing:
        raise HTTPException(status_code=400, detail="Employee number already exists")
    
    # Check if CIN already exists
    existing_cin = await db.employees.find_one({"cin_number": employee_data.cin_number})
    if existing_cin:
        raise HTTPException(status_code=400, detail="CIN number already registered")
    
    employee = Employee(**employee_data.model_dump())
    employee_dict = employee.model_dump()
    employee_dict['created_at'] = employee_dict['created_at'].isoformat()
    employee_dict['date_of_birth'] = employee_dict['date_of_birth'].isoformat()
    employee_dict['hire_date'] = employee_dict['hire_date'].isoformat()
    if employee_dict.get('contract_end_date'):
        employee_dict['contract_end_date'] = employee_dict['contract_end_date'].isoformat()
    
    await db.employees.insert_one(employee_dict)
    return employee

@router.put("/employees/{employee_id}")
async def update_employee(
    employee_id: str,
    employee_data: dict,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    if current_user.get("role") not in ["superadmin", "admin"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    employee_data["updated_at"] = datetime.now(timezone.utc).isoformat()
    
    result = await db.employees.update_one(
        {"id": employee_id},
        {"$set": employee_data}
    )
    
    if result.modified_count == 0:
        raise HTTPException(status_code=404, detail="Employee not found")
    
    return {"message": "Employee updated successfully"}

@router.patch("/employees/{employee_id}/terminate")
async def terminate_employee(
    employee_id: str,
    termination_data: dict,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    if current_user.get("role") not in ["superadmin", "admin"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    update_data = {
        "status": EmployeeStatus.TERMINATED.value,
        "termination_date": termination_data.get("termination_date", datetime.now(timezone.utc).isoformat()),
        "termination_reason": termination_data.get("termination_reason"),
        "updated_at": datetime.now(timezone.utc).isoformat()
    }
    
    result = await db.employees.update_one(
        {"id": employee_id},
        {"$set": update_data}
    )
    
    if result.modified_count == 0:
        raise HTTPException(status_code=404, detail="Employee not found")
    
    return {"message": "Employee terminated successfully"}

# ==================== POINTAGE ====================
@router.get("/attendance", response_model=List[Attendance])
async def get_attendance(
    employee_id: Optional[str] = None,
    date: Optional[str] = None,
    start_date: Optional[str] = None,
    end_date: Optional[str] = None,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    query = {}
    if employee_id:
        query["employee_id"] = employee_id
    if date:
        query["date"] = {"$gte": date, "$lt": date}
    elif start_date and end_date:
        query["date"] = {"$gte": start_date, "$lte": end_date}
    
    attendances = await db.attendances.find(query, {"_id": 0}).sort("date", -1).to_list(None)
    
    for att in attendances:
        if isinstance(att.get('date'), str):
            att['date'] = datetime.fromisoformat(att['date'])
        if att.get('check_in') and isinstance(att['check_in'], str):
            att['check_in'] = datetime.fromisoformat(att['check_in'])
        if att.get('check_out') and isinstance(att['check_out'], str):
            att['check_out'] = datetime.fromisoformat(att['check_out'])
    
    return attendances

@router.post("/attendance/check-in")
async def check_in(
    employee_id: str,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    now = datetime.now(timezone.utc)
    today = now.date()
    
    # Check if already checked in today
    existing = await db.attendances.find_one({
        "employee_id": employee_id,
        "date": {"$gte": today.isoformat(), "$lt": (today.isoformat() + "T23:59:59")}
    })
    
    if existing and existing.get("check_in"):
        raise HTTPException(status_code=400, detail="Already checked in today")
    
    if existing:
        # Update existing record
        await db.attendances.update_one(
            {"id": existing["id"]},
            {"$set": {"check_in": now.isoformat()}}
        )
        return {"message": "Check-in recorded", "time": now.isoformat()}
    else:
        # Create new attendance record
        attendance = Attendance(
            employee_id=employee_id,
            date=now,
            check_in=now
        )
        attendance_dict = attendance.model_dump()
        attendance_dict['created_at'] = attendance_dict['created_at'].isoformat()
        attendance_dict['date'] = attendance_dict['date'].isoformat()
        attendance_dict['check_in'] = attendance_dict['check_in'].isoformat()
        
        await db.attendances.insert_one(attendance_dict)
        return {"message": "Check-in recorded", "time": now.isoformat()}

@router.post("/attendance/check-out")
async def check_out(
    employee_id: str,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    now = datetime.now(timezone.utc)
    today = now.date()
    
    # Find today's attendance
    attendance = await db.attendances.find_one({
        "employee_id": employee_id,
        "date": {"$gte": today.isoformat(), "$lt": (today.isoformat() + "T23:59:59")}
    })
    
    if not attendance or not attendance.get("check_in"):
        raise HTTPException(status_code=400, detail="No check-in found for today")
    
    if attendance.get("check_out"):
        raise HTTPException(status_code=400, detail="Already checked out")
    
    check_in_time = datetime.fromisoformat(attendance["check_in"])
    worked_hours = (now - check_in_time).total_seconds() / 3600
    
    # Calculate overtime (assuming 8 hours work day)
    overtime_hours = max(0, worked_hours - 8)
    
    await db.attendances.update_one(
        {"id": attendance["id"]},
        {
            "$set": {
                "check_out": now.isoformat(),
                "worked_hours": round(worked_hours, 2),
                "overtime_hours": round(overtime_hours, 2)
            }
        }
    )
    
    return {
        "message": "Check-out recorded",
        "time": now.isoformat(),
        "worked_hours": round(worked_hours, 2),
        "overtime_hours": round(overtime_hours, 2)
    }

# ==================== CONGÉS/ABSENCES ====================
@router.get("/leave-requests", response_model=List[LeaveRequest])
async def get_leave_requests(
    employee_id: Optional[str] = None,
    status: Optional[str] = None,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    query = {}
    if employee_id:
        query["employee_id"] = employee_id
    if status:
        query["status"] = status
    
    requests = await db.leave_requests.find(query, {"_id": 0}).sort("start_date", -1).to_list(None)
    
    for req in requests:
        if isinstance(req.get('start_date'), str):
            req['start_date'] = datetime.fromisoformat(req['start_date'])
        if isinstance(req.get('end_date'), str):
            req['end_date'] = datetime.fromisoformat(req['end_date'])
        if req.get('approval_date') and isinstance(req['approval_date'], str):
            req['approval_date'] = datetime.fromisoformat(req['approval_date'])
    
    return requests

@router.post("/leave-requests", response_model=LeaveRequest)
async def create_leave_request(
    request_data: dict,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    # Calculate days count
    start = datetime.fromisoformat(request_data["start_date"])
    end = datetime.fromisoformat(request_data["end_date"])
    days_count = (end - start).days + 1
    request_data["days_count"] = days_count
    
    # Get employee name
    employee = await db.employees.find_one({"id": request_data["employee_id"]})
    if not employee:
        raise HTTPException(status_code=404, detail="Employee not found")
    
    request_data["employee_name"] = f"{employee['first_name']} {employee['last_name']}"
    
    leave_request = LeaveRequest(**request_data)
    leave_dict = leave_request.model_dump()
    leave_dict['created_at'] = leave_dict['created_at'].isoformat()
    leave_dict['start_date'] = leave_dict['start_date'].isoformat()
    leave_dict['end_date'] = leave_dict['end_date'].isoformat()
    
    await db.leave_requests.insert_one(leave_dict)
    return leave_request

@router.patch("/leave-requests/{request_id}/approve")
async def approve_leave_request(
    request_id: str,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    if current_user.get("role") not in ["superadmin", "admin", "manager"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    result = await db.leave_requests.update_one(
        {"id": request_id},
        {
            "$set": {
                "status": AbsenceStatus.APPROVED.value,
                "approved_by": current_user.get("id"),
                "approval_date": datetime.now(timezone.utc).isoformat(),
                "updated_at": datetime.now(timezone.utc).isoformat()
            }
        }
    )
    
    if result.modified_count == 0:
        raise HTTPException(status_code=404, detail="Leave request not found")
    
    return {"message": "Leave request approved"}

@router.patch("/leave-requests/{request_id}/reject")
async def reject_leave_request(
    request_id: str,
    rejection_reason: str,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    if current_user.get("role") not in ["superadmin", "admin", "manager"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    result = await db.leave_requests.update_one(
        {"id": request_id},
        {
            "$set": {
                "status": AbsenceStatus.REJECTED.value,
                "rejection_reason": rejection_reason,
                "approved_by": current_user.get("id"),
                "approval_date": datetime.now(timezone.utc).isoformat(),
                "updated_at": datetime.now(timezone.utc).isoformat()
            }
        }
    )
    
    if result.modified_count == 0:
        raise HTTPException(status_code=404, detail="Leave request not found")
    
    return {"message": "Leave request rejected"}

# ==================== BULLETINS DE PAIE ====================
@router.get("/payslips", response_model=List[Payslip])
async def get_payslips(
    employee_id: Optional[str] = None,
    period_month: Optional[int] = None,
    period_year: Optional[int] = None,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    if current_user.get("role") not in ["superadmin", "admin", "manager"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    query = {}
    if employee_id:
        query["employee_id"] = employee_id
    if period_month:
        query["period_month"] = period_month
    if period_year:
        query["period_year"] = period_year
    
    payslips = await db.payslips.find(query, {"_id": 0}).sort([("period_year", -1), ("period_month", -1)]).to_list(None)
    
    for payslip in payslips:
        if isinstance(payslip.get('payment_date'), str):
            payslip['payment_date'] = datetime.fromisoformat(payslip['payment_date'])
    
    return payslips

@router.post("/payslips/generate")
async def generate_payslip(
    employee_id: str,
    period_month: int,
    period_year: int,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    """Generate payslip for an employee"""
    if current_user.get("role") not in ["superadmin", "admin"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    # Check if payslip already exists
    existing = await db.payslips.find_one({
        "employee_id": employee_id,
        "period_month": period_month,
        "period_year": period_year
    })
    if existing:
        raise HTTPException(status_code=400, detail="Payslip already exists for this period")
    
    # Get employee
    employee = await db.employees.find_one({"id": employee_id})
    if not employee:
        raise HTTPException(status_code=404, detail="Employee not found")
    
    base_salary = employee.get("base_salary", 0)
    
    # Calculate gross salary (base + allowances)
    # For now, just use base salary
    gross_salary = base_salary
    
    # Calculate social charges (CNSS)
    cnss_employee = gross_salary * 0.09  # 9%
    cnss_employer = gross_salary * 0.26  # 26%
    
    # Calculate taxable salary
    taxable_salary = gross_salary - cnss_employee
    
    # Calculate IRG (simplified - would need proper tax brackets)
    if taxable_salary <= 30000:
        irg_amount = 0
    elif taxable_salary <= 120000:
        irg_amount = (taxable_salary - 30000) * 0.20
    else:
        irg_amount = (120000 - 30000) * 0.20 + (taxable_salary - 120000) * 0.35
    
    # Calculate net salary
    total_deductions = cnss_employee + irg_amount
    net_salary = gross_salary - total_deductions
    
    payslip = Payslip(
        employee_id=employee_id,
        employee_name=f"{employee['first_name']} {employee['last_name']}",
        employee_number=employee.get("employee_number", ""),
        period_month=period_month,
        period_year=period_year,
        base_salary=base_salary,
        gross_salary=gross_salary,
        cnss_employee=cnss_employee,
        cnss_employer=cnss_employer,
        taxable_salary=taxable_salary,
        irg_amount=irg_amount,
        total_deductions=total_deductions,
        net_salary=net_salary,
        payment_date=datetime.now(timezone.utc)
    )
    
    payslip_dict = payslip.model_dump()
    payslip_dict['created_at'] = payslip_dict['created_at'].isoformat()
    payslip_dict['payment_date'] = payslip_dict['payment_date'].isoformat()
    
    await db.payslips.insert_one(payslip_dict)
    return payslip

# ==================== FORMATIONS ====================
@router.get("/trainings", response_model=List[Training])
async def get_trainings(
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    trainings = await db.trainings.find({}, {"_id": 0}).sort("start_date", -1).to_list(None)
    
    for training in trainings:
        if isinstance(training.get('start_date'), str):
            training['start_date'] = datetime.fromisoformat(training['start_date'])
        if isinstance(training.get('end_date'), str):
            training['end_date'] = datetime.fromisoformat(training['end_date'])
    
    return trainings

@router.post("/trainings", response_model=Training)
async def create_training(
    training_data: dict,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    if current_user.get("role") not in ["superadmin", "admin"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    training = Training(**training_data)
    training_dict = training.model_dump()
    training_dict['created_at'] = training_dict['created_at'].isoformat()
    training_dict['start_date'] = training_dict['start_date'].isoformat()
    training_dict['end_date'] = training_dict['end_date'].isoformat()
    
    await db.trainings.insert_one(training_dict)
    return training

# ==================== DASHBOARD RH ====================
@router.get("/dashboard")
async def get_hr_dashboard(
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    if current_user.get("role") not in ["superadmin", "admin", "manager"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    # Total employees
    total_employees = await db.employees.count_documents({"status": EmployeeStatus.ACTIVE.value})
    
    # By department
    departments = await db.employees.distinct("department", {"status": EmployeeStatus.ACTIVE.value})
    
    # Pending leave requests
    pending_leaves = await db.leave_requests.count_documents({"status": AbsenceStatus.PENDING.value})
    
    # Today's attendance
    today = datetime.now(timezone.utc).date()
    today_attendance = await db.attendances.count_documents({
        "date": {"$gte": today.isoformat(), "$lt": (today.isoformat() + "T23:59:59")},
        "check_in": {"$ne": None}
    })
    
    # Upcoming trainings
    upcoming_trainings = await db.trainings.count_documents({
        "start_date": {"$gte": datetime.now(timezone.utc).isoformat()}
    })
    
    return {
        "total_employees": total_employees,
        "departments_count": len(departments),
        "pending_leave_requests": pending_leaves,
        "today_attendance": today_attendance,
        "upcoming_trainings": upcoming_trainings
    }
