from fastapi import APIRouter, HTTPException, Depends
from typing import List, Optional
from datetime import datetime, timezone, timedelta
from models import (
    TVADeclaration, IBSDeclaration, TAPDeclaration,
    FiscalDeadline, TaxType, TaxDeclarationStatus
)
from dependencies import get_current_user, get_db
from motor.motor_asyncio import AsyncIOMotorDatabase

router = APIRouter(prefix="/fiscal", tags=["fiscal"])

# ==================== DÉCLARATION TVA ====================
@router.get("/tva/declarations", response_model=List[TVADeclaration])
async def get_tva_declarations(
    year: Optional[int] = None,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    if current_user.get("role") not in ["superadmin", "admin", "manager"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    query = {}
    if year:
        query["period_year"] = year
    
    declarations = await db.tva_declarations.find(query, {"_id": 0}).sort([("period_year", -1), ("period_month", -1)]).to_list(None)
    
    for decl in declarations:
        if isinstance(decl.get('declaration_date'), str):
            decl['declaration_date'] = datetime.fromisoformat(decl['declaration_date'])
        if isinstance(decl.get('due_date'), str):
            decl['due_date'] = datetime.fromisoformat(decl['due_date'])
    
    return declarations

@router.post("/tva/declarations", response_model=TVADeclaration)
async def create_tva_declaration(
    declaration_data: dict,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    if current_user.get("role") not in ["superadmin", "admin", "manager"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    # Check if declaration already exists for this period
    existing = await db.tva_declarations.find_one({
        "period_month": declaration_data["period_month"],
        "period_year": declaration_data["period_year"]
    })
    if existing:
        raise HTTPException(status_code=400, detail="TVA declaration already exists for this period")
    
    # Calculate TVA collectée (19% des ventes HT)
    declaration_data["tva_collected"] = declaration_data["sales_ht"] * 0.19
    
    # Calculate TVA récupérable (19% des achats HT)
    declaration_data["tva_recoverable"] = declaration_data["purchases_ht"] * 0.19
    
    # Calculate TVA due or credit
    diff = declaration_data["tva_collected"] - declaration_data["tva_recoverable"]
    if diff > 0:
        declaration_data["tva_due"] = diff
        declaration_data["tva_credit"] = 0
    else:
        declaration_data["tva_due"] = 0
        declaration_data["tva_credit"] = abs(diff)
    
    # Set due date (20th of next month)
    year = declaration_data["period_year"]
    month = declaration_data["period_month"]
    if month == 12:
        due_year = year + 1
        due_month = 1
    else:
        due_year = year
        due_month = month + 1
    
    declaration_data["due_date"] = datetime(due_year, due_month, 20)
    declaration_data["declaration_date"] = datetime.now(timezone.utc)
    
    declaration = TVADeclaration(**declaration_data)
    declaration_dict = declaration.model_dump()
    declaration_dict['created_at'] = declaration_dict['created_at'].isoformat()
    declaration_dict['declaration_date'] = declaration_dict['declaration_date'].isoformat()
    declaration_dict['due_date'] = declaration_dict['due_date'].isoformat()
    
    await db.tva_declarations.insert_one(declaration_dict)
    return declaration

@router.patch("/tva/declarations/{declaration_id}/submit")
async def submit_tva_declaration(
    declaration_id: str,
    payment_reference: Optional[str] = None,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    if current_user.get("role") not in ["superadmin", "admin", "manager"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    update_data = {
        "status": TaxDeclarationStatus.SUBMITTED.value,
        "submitted_by": current_user.get("id"),
        "updated_at": datetime.now(timezone.utc).isoformat()
    }
    
    if payment_reference:
        update_data["payment_reference"] = payment_reference
        update_data["status"] = TaxDeclarationStatus.PAID.value
    
    result = await db.tva_declarations.update_one(
        {"id": declaration_id},
        {"$set": update_data}
    )
    
    if result.modified_count == 0:
        raise HTTPException(status_code=404, detail="TVA declaration not found")
    
    return {"message": "TVA declaration submitted successfully"}

# ==================== DÉCLARATION IBS ====================
@router.get("/ibs/declarations", response_model=List[IBSDeclaration])
async def get_ibs_declarations(
    fiscal_year: Optional[int] = None,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    if current_user.get("role") not in ["superadmin", "admin", "manager"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    query = {}
    if fiscal_year:
        query["fiscal_year"] = fiscal_year
    
    declarations = await db.ibs_declarations.find(query, {"_id": 0}).sort([("fiscal_year", -1), ("quarter", 1)]).to_list(None)
    
    for decl in declarations:
        if isinstance(decl.get('declaration_date'), str):
            decl['declaration_date'] = datetime.fromisoformat(decl['declaration_date'])
        if isinstance(decl.get('due_date'), str):
            decl['due_date'] = datetime.fromisoformat(decl['due_date'])
    
    return declarations

@router.post("/ibs/declarations", response_model=IBSDeclaration)
async def create_ibs_declaration(
    declaration_data: dict,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    if current_user.get("role") not in ["superadmin", "admin", "manager"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    # Calculate taxable result
    taxable_result = (
        declaration_data["accounting_result"] +
        declaration_data.get("reintegrations", 0) -
        declaration_data.get("deductions", 0)
    )
    declaration_data["taxable_result"] = taxable_result
    
    # Calculate IBS amount (26%)
    ibs_rate = declaration_data.get("ibs_rate", 26.0)
    declaration_data["ibs_amount"] = taxable_result * (ibs_rate / 100)
    
    # Set due dates based on type
    year = declaration_data["fiscal_year"]
    if declaration_data["declaration_type"] == "acompte":
        quarter = declaration_data["quarter"]
        # Acomptes due on 20/03, 20/06, 20/09, 20/12
        due_month = quarter * 3
        declaration_data["due_date"] = datetime(year, due_month, 20)
    else:  # annual
        # Annual declaration due on 30/04 of next year
        declaration_data["due_date"] = datetime(year + 1, 4, 30)
    
    declaration_data["declaration_date"] = datetime.now(timezone.utc)
    
    declaration = IBSDeclaration(**declaration_data)
    declaration_dict = declaration.model_dump()
    declaration_dict['created_at'] = declaration_dict['created_at'].isoformat()
    declaration_dict['declaration_date'] = declaration_dict['declaration_date'].isoformat()
    declaration_dict['due_date'] = declaration_dict['due_date'].isoformat()
    
    await db.ibs_declarations.insert_one(declaration_dict)
    return declaration

# ==================== DÉCLARATION TAP ====================
@router.get("/tap/declarations", response_model=List[TAPDeclaration])
async def get_tap_declarations(
    year: Optional[int] = None,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    if current_user.get("role") not in ["superadmin", "admin", "manager"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    query = {}
    if year:
        query["period_year"] = year
    
    declarations = await db.tap_declarations.find(query, {"_id": 0}).sort([("period_year", -1), ("period_month", -1)]).to_list(None)
    
    for decl in declarations:
        if isinstance(decl.get('declaration_date'), str):
            decl['declaration_date'] = datetime.fromisoformat(decl['declaration_date'])
        if isinstance(decl.get('due_date'), str):
            decl['due_date'] = datetime.fromisoformat(decl['due_date'])
    
    return declarations

@router.post("/tap/declarations", response_model=TAPDeclaration)
async def create_tap_declaration(
    declaration_data: dict,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    if current_user.get("role") not in ["superadmin", "admin", "manager"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    # Check if declaration already exists
    existing = await db.tap_declarations.find_one({
        "period_month": declaration_data["period_month"],
        "period_year": declaration_data["period_year"]
    })
    if existing:
        raise HTTPException(status_code=400, detail="TAP declaration already exists for this period")
    
    # Calculate TAP (2% of turnover HT)
    tap_rate = declaration_data.get("tap_rate", 2.0)
    declaration_data["tap_amount"] = declaration_data["turnover_ht"] * (tap_rate / 100)
    
    # Set due date (20th of next month)
    year = declaration_data["period_year"]
    month = declaration_data["period_month"]
    if month == 12:
        due_year = year + 1
        due_month = 1
    else:
        due_year = year
        due_month = month + 1
    
    declaration_data["due_date"] = datetime(due_year, due_month, 20)
    declaration_data["declaration_date"] = datetime.now(timezone.utc)
    
    declaration = TAPDeclaration(**declaration_data)
    declaration_dict = declaration.model_dump()
    declaration_dict['created_at'] = declaration_dict['created_at'].isoformat()
    declaration_dict['declaration_date'] = declaration_dict['declaration_date'].isoformat()
    declaration_dict['due_date'] = declaration_dict['due_date'].isoformat()
    
    await db.tap_declarations.insert_one(declaration_dict)
    return declaration

# ==================== CALENDRIER FISCAL ====================
@router.get("/deadlines", response_model=List[FiscalDeadline])
async def get_fiscal_deadlines(
    year: Optional[int] = None,
    tax_type: Optional[str] = None,
    is_completed: Optional[bool] = None,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    if current_user.get("role") not in ["superadmin", "admin", "manager"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    query = {}
    if year:
        query["period_year"] = year
    if tax_type:
        query["tax_type"] = tax_type
    if is_completed is not None:
        query["is_completed"] = is_completed
    
    deadlines = await db.fiscal_deadlines.find(query, {"_id": 0}).sort("due_date", 1).to_list(None)
    
    for deadline in deadlines:
        if isinstance(deadline.get('due_date'), str):
            deadline['due_date'] = datetime.fromisoformat(deadline['due_date'])
    
    return deadlines

@router.post("/deadlines/generate/{year}")
async def generate_fiscal_deadlines(
    year: int,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    """Generate fiscal deadlines for a given year"""
    if current_user.get("role") not in ["superadmin", "admin"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    # Check if deadlines already exist
    existing_count = await db.fiscal_deadlines.count_documents({"period_year": year})
    if existing_count > 0:
        raise HTTPException(status_code=400, detail=f"Fiscal deadlines already exist for year {year}")
    
    deadlines = []
    
    # TVA declarations (monthly, 20th of next month)
    for month in range(1, 13):
        due_month = month + 1 if month < 12 else 1
        due_year = year if month < 12 else year + 1
        
        deadline = FiscalDeadline(
            tax_type=TaxType.TVA,
            name=f"Déclaration TVA {month:02d}/{year}",
            due_date=datetime(due_year, due_month, 20),
            period_month=month,
            period_year=year
        )
        deadlines.append(deadline.model_dump())
    
    # TAP declarations (monthly, 20th of next month)
    for month in range(1, 13):
        due_month = month + 1 if month < 12 else 1
        due_year = year if month < 12 else year + 1
        
        deadline = FiscalDeadline(
            tax_type=TaxType.TAP,
            name=f"Déclaration TAP {month:02d}/{year}",
            due_date=datetime(due_year, due_month, 20),
            period_month=month,
            period_year=year
        )
        deadlines.append(deadline.model_dump())
    
    # IBS acomptes (quarterly)
    for quarter in [1, 2, 3, 4]:
        month = quarter * 3
        deadline = FiscalDeadline(
            tax_type=TaxType.IBS,
            name=f"Acompte IBS T{quarter} {year}",
            due_date=datetime(year, month, 20),
            period_year=year
        )
        deadlines.append(deadline.model_dump())
    
    # IBS annual declaration
    deadline = FiscalDeadline(
        tax_type=TaxType.IBS,
        name=f"Déclaration IBS annuelle {year}",
        due_date=datetime(year + 1, 4, 30),
        period_year=year
    )
    deadlines.append(deadline.model_dump())
    
    # Convert datetime objects to ISO strings
    for deadline in deadlines:
        deadline['created_at'] = deadline['created_at'].isoformat()
        deadline['due_date'] = deadline['due_date'].isoformat()
    
    await db.fiscal_deadlines.insert_many(deadlines)
    
    return {"message": f"Generated {len(deadlines)} fiscal deadlines for year {year}"}

@router.patch("/deadlines/{deadline_id}/complete")
async def mark_deadline_complete(
    deadline_id: str,
    declaration_id: str,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    if current_user.get("role") not in ["superadmin", "admin", "manager"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    result = await db.fiscal_deadlines.update_one(
        {"id": deadline_id},
        {
            "$set": {
                "is_completed": True,
                "declaration_id": declaration_id,
                "updated_at": datetime.now(timezone.utc).isoformat()
            }
        }
    )
    
    if result.modified_count == 0:
        raise HTTPException(status_code=404, detail="Fiscal deadline not found")
    
    return {"message": "Deadline marked as complete"}

# ==================== DASHBOARD FISCAL ====================
@router.get("/dashboard")
async def get_fiscal_dashboard(
    year: Optional[int] = None,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    if current_user.get("role") not in ["superadmin", "admin", "manager"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    if not year:
        year = datetime.now().year
    
    # Upcoming deadlines (next 30 days)
    now = datetime.now(timezone.utc)
    next_month = now + timedelta(days=30)
    
    upcoming = await db.fiscal_deadlines.find({
        "due_date": {"$gte": now.isoformat(), "$lte": next_month.isoformat()},
        "is_completed": False
    }, {"_id": 0}).sort("due_date", 1).to_list(None)
    
    # Overdue deadlines
    overdue = await db.fiscal_deadlines.find({
        "due_date": {"$lt": now.isoformat()},
        "is_completed": False
    }, {"_id": 0}).to_list(None)
    
    # TVA statistics for current year
    tva_declarations = await db.tva_declarations.find(
        {"period_year": year},
        {"_id": 0}
    ).to_list(None)
    
    total_tva_collected = sum(d.get("tva_collected", 0) for d in tva_declarations)
    total_tva_recoverable = sum(d.get("tva_recoverable", 0) for d in tva_declarations)
    total_tva_due = sum(d.get("tva_due", 0) for d in tva_declarations)
    
    # Compliance rate
    total_deadlines = await db.fiscal_deadlines.count_documents({"period_year": year})
    completed_deadlines = await db.fiscal_deadlines.count_documents({
        "period_year": year,
        "is_completed": True
    })
    
    compliance_rate = (completed_deadlines / total_deadlines * 100) if total_deadlines > 0 else 0
    
    return {
        "upcoming_deadlines": len(upcoming),
        "overdue_deadlines": len(overdue),
        "compliance_rate": round(compliance_rate, 2),
        "total_tva_collected": total_tva_collected,
        "total_tva_recoverable": total_tva_recoverable,
        "total_tva_due": total_tva_due,
        "tva_declarations_count": len(tva_declarations)
    }
