from fastapi import APIRouter, HTTPException, Depends
from typing import List, Optional
from datetime import datetime, timezone
from models import (
    Account, AccountCreate, AccountType,
    JournalEntry, JournalEntryCreate, JournalType, TransactionStatus,
    BankAccount, BankTransaction,
    CashRegister, CashTransaction,
    CustomerPayment, SupplierPayment, PaymentMethod,
    Budget, Asset, AssetCategory
)
from dependencies import get_current_user, get_db
from motor.motor_asyncio import AsyncIOMotorDatabase

router = APIRouter(prefix="/finance", tags=["finance"])

# ==================== COMPTES COMPTABLES ====================
@router.get("/accounts", response_model=List[Account])
async def get_accounts(
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    if current_user.get("role") not in ["superadmin", "admin", "manager"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    accounts = await db.accounts.find({}, {"_id": 0}).sort("code", 1).to_list(None)
    return accounts

@router.post("/accounts", response_model=Account)
async def create_account(
    account_data: AccountCreate,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    if current_user.get("role") not in ["superadmin", "admin"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    # Check if account code already exists
    existing = await db.accounts.find_one({"code": account_data.code})
    if existing:
        raise HTTPException(status_code=400, detail="Account code already exists")
    
    account = Account(**account_data.model_dump())
    account_dict = account.model_dump()
    account_dict['created_at'] = account_dict['created_at'].isoformat()
    
    await db.accounts.insert_one(account_dict)
    return account

# ==================== ÉCRITURES COMPTABLES ====================
@router.get("/journal-entries", response_model=List[JournalEntry])
async def get_journal_entries(
    journal_type: Optional[str] = None,
    start_date: Optional[str] = None,
    end_date: Optional[str] = None,
    status: Optional[str] = None,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    if current_user.get("role") not in ["superadmin", "admin", "manager"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    query = {}
    if journal_type:
        query["journal_type"] = journal_type
    if status:
        query["status"] = status
    if start_date and end_date:
        query["date"] = {
            "$gte": start_date,
            "$lte": end_date
        }
    
    entries = await db.journal_entries.find(query, {"_id": 0}).sort("date", -1).to_list(None)
    
    for entry in entries:
        if isinstance(entry.get('date'), str):
            entry['date'] = datetime.fromisoformat(entry['date'])
    
    return entries

@router.post("/journal-entries", response_model=JournalEntry)
async def create_journal_entry(
    entry_data: JournalEntryCreate,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    if current_user.get("role") not in ["superadmin", "admin", "manager"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    # Calculate totals
    total_debit = sum(line.debit for line in entry_data.lines)
    total_credit = sum(line.credit for line in entry_data.lines)
    
    if abs(total_debit - total_credit) > 0.01:
        raise HTTPException(status_code=400, detail="Debit and credit must be equal")
    
    # Generate entry number
    count = await db.journal_entries.count_documents({})
    entry_number = f"JE-{count + 1:06d}"
    
    entry = JournalEntry(
        entry_number=entry_number,
        journal_type=entry_data.journal_type,
        date=entry_data.date,
        description=entry_data.description,
        reference=entry_data.reference,
        lines=entry_data.lines,
        total_debit=total_debit,
        total_credit=total_credit,
        status=TransactionStatus.DRAFT
    )
    
    entry_dict = entry.model_dump()
    entry_dict['created_at'] = entry_dict['created_at'].isoformat()
    entry_dict['date'] = entry_dict['date'].isoformat()
    
    await db.journal_entries.insert_one(entry_dict)
    return entry

@router.patch("/journal-entries/{entry_id}/post")
async def post_journal_entry(
    entry_id: str,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    if current_user.get("role") not in ["superadmin", "admin", "manager"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    result = await db.journal_entries.update_one(
        {"id": entry_id},
        {
            "$set": {
                "status": TransactionStatus.POSTED.value,
                "posted_by": current_user.get("id"),
                "updated_at": datetime.now(timezone.utc).isoformat()
            }
        }
    )
    
    if result.modified_count == 0:
        raise HTTPException(status_code=404, detail="Journal entry not found")
    
    # Update account balances
    entry = await db.journal_entries.find_one({"id": entry_id})
    for line in entry["lines"]:
        await db.accounts.update_one(
            {"code": line["account_code"]},
            {"$inc": {"balance": line["debit"] - line["credit"]}}
        )
    
    return {"message": "Journal entry posted successfully"}

# ==================== BANQUES ====================
@router.get("/banks", response_model=List[BankAccount])
async def get_bank_accounts(
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    if current_user.get("role") not in ["superadmin", "admin", "manager"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    banks = await db.bank_accounts.find({}, {"_id": 0}).to_list(None)
    return banks

@router.post("/banks", response_model=BankAccount)
async def create_bank_account(
    bank_data: dict,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    if current_user.get("role") not in ["superadmin", "admin"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    bank = BankAccount(**bank_data)
    bank_dict = bank.model_dump()
    bank_dict['created_at'] = bank_dict['created_at'].isoformat()
    
    await db.bank_accounts.insert_one(bank_dict)
    return bank

@router.get("/banks/{bank_id}/transactions", response_model=List[BankTransaction])
async def get_bank_transactions(
    bank_id: str,
    start_date: Optional[str] = None,
    end_date: Optional[str] = None,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    if current_user.get("role") not in ["superadmin", "admin", "manager"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    query = {"bank_account_id": bank_id}
    if start_date and end_date:
        query["date"] = {"$gte": start_date, "$lte": end_date}
    
    transactions = await db.bank_transactions.find(query, {"_id": 0}).sort("date", -1).to_list(None)
    
    for trans in transactions:
        if isinstance(trans.get('date'), str):
            trans['date'] = datetime.fromisoformat(trans['date'])
    
    return transactions

# ==================== CAISSES ====================
@router.get("/cash-registers", response_model=List[CashRegister])
async def get_cash_registers(
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    cash_registers = await db.cash_registers.find({}, {"_id": 0}).to_list(None)
    return cash_registers

@router.post("/cash-registers", response_model=CashRegister)
async def create_cash_register(
    cash_data: dict,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    if current_user.get("role") not in ["superadmin", "admin"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    cash_register = CashRegister(**cash_data)
    cash_dict = cash_register.model_dump()
    cash_dict['created_at'] = cash_dict['created_at'].isoformat()
    
    await db.cash_registers.insert_one(cash_dict)
    return cash_register

@router.get("/cash-registers/{register_id}/transactions", response_model=List[CashTransaction])
async def get_cash_transactions(
    register_id: str,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    transactions = await db.cash_transactions.find(
        {"cash_register_id": register_id},
        {"_id": 0}
    ).sort("date", -1).to_list(None)
    
    for trans in transactions:
        if isinstance(trans.get('date'), str):
            trans['date'] = datetime.fromisoformat(trans['date'])
    
    return transactions

# ==================== ENCAISSEMENTS/DÉCAISSEMENTS ====================
@router.post("/customer-payments", response_model=CustomerPayment)
async def create_customer_payment(
    payment_data: dict,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    if current_user.get("role") not in ["superadmin", "admin", "manager"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    payment = CustomerPayment(**payment_data)
    payment_dict = payment.model_dump()
    payment_dict['created_at'] = payment_dict['created_at'].isoformat()
    payment_dict['payment_date'] = payment_dict['payment_date'].isoformat()
    
    await db.customer_payments.insert_one(payment_dict)
    
    # Update bank or cash balance
    if payment_data.get('bank_account_id'):
        await db.bank_accounts.update_one(
            {"id": payment_data['bank_account_id']},
            {"$inc": {"balance": payment_data['amount']}}
        )
    elif payment_data.get('cash_register_id'):
        await db.cash_registers.update_one(
            {"id": payment_data['cash_register_id']},
            {"$inc": {"balance": payment_data['amount']}}
        )
    
    return payment

@router.get("/customer-payments", response_model=List[CustomerPayment])
async def get_customer_payments(
    customer_id: Optional[str] = None,
    start_date: Optional[str] = None,
    end_date: Optional[str] = None,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    if current_user.get("role") not in ["superadmin", "admin", "manager"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    query = {}
    if customer_id:
        query["customer_id"] = customer_id
    if start_date and end_date:
        query["payment_date"] = {"$gte": start_date, "$lte": end_date}
    
    payments = await db.customer_payments.find(query, {"_id": 0}).sort("payment_date", -1).to_list(None)
    
    for payment in payments:
        if isinstance(payment.get('payment_date'), str):
            payment['payment_date'] = datetime.fromisoformat(payment['payment_date'])
    
    return payments

# ==================== BUDGETS ====================
@router.get("/budgets", response_model=List[Budget])
async def get_budgets(
    fiscal_year: Optional[int] = None,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    if current_user.get("role") not in ["superadmin", "admin", "manager"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    query = {}
    if fiscal_year:
        query["fiscal_year"] = fiscal_year
    
    budgets = await db.budgets.find(query, {"_id": 0}).to_list(None)
    
    for budget in budgets:
        if isinstance(budget.get('start_date'), str):
            budget['start_date'] = datetime.fromisoformat(budget['start_date'])
        if isinstance(budget.get('end_date'), str):
            budget['end_date'] = datetime.fromisoformat(budget['end_date'])
    
    return budgets

# ==================== IMMOBILISATIONS ====================
@router.get("/assets", response_model=List[Asset])
async def get_assets(
    category: Optional[str] = None,
    is_active: Optional[bool] = None,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    if current_user.get("role") not in ["superadmin", "admin", "manager"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    query = {}
    if category:
        query["category"] = category
    if is_active is not None:
        query["is_active"] = is_active
    
    assets = await db.assets.find(query, {"_id": 0}).to_list(None)
    
    for asset in assets:
        if isinstance(asset.get('purchase_date'), str):
            asset['purchase_date'] = datetime.fromisoformat(asset['purchase_date'])
    
    return assets

@router.post("/assets", response_model=Asset)
async def create_asset(
    asset_data: dict,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    if current_user.get("role") not in ["superadmin", "admin"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    # Calculate net book value
    asset_data['net_book_value'] = asset_data['purchase_value'] - asset_data.get('accumulated_depreciation', 0)
    
    asset = Asset(**asset_data)
    asset_dict = asset.model_dump()
    asset_dict['created_at'] = asset_dict['created_at'].isoformat()
    asset_dict['purchase_date'] = asset_dict['purchase_date'].isoformat()
    
    await db.assets.insert_one(asset_dict)
    return asset

# ==================== DASHBOARD FINANCE ====================
@router.get("/dashboard")
async def get_finance_dashboard(
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    if current_user.get("role") not in ["superadmin", "admin", "manager"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    # Total bank balances
    banks = await db.bank_accounts.find({"is_active": True}, {"_id": 0}).to_list(None)
    total_bank_balance = sum(b.get("balance", 0) for b in banks)
    
    # Total cash balances
    cash_registers = await db.cash_registers.find({"is_active": True}, {"_id": 0}).to_list(None)
    total_cash_balance = sum(c.get("balance", 0) for c in cash_registers)
    
    # Total treasury
    total_treasury = total_bank_balance + total_cash_balance
    
    # Pending customer payments (créances)
    # This would need invoice data
    
    # Pending supplier payments (dettes)
    # This would need supplier invoice data
    
    return {
        "total_bank_balance": total_bank_balance,
        "total_cash_balance": total_cash_balance,
        "total_treasury": total_treasury,
        "bank_accounts_count": len(banks),
        "cash_registers_count": len(cash_registers)
    }
