"""
Module IA - Intelligence Artificielle et Détection d'Anomalies
Utilise OpenAI GPT-4 via la clé Emergent universelle
"""
from fastapi import APIRouter, HTTPException, Depends
from typing import List, Dict, Optional
from datetime import datetime, timezone, timedelta
from dependencies import get_current_user, get_db
from motor.motor_asyncio import AsyncIOMotorDatabase
import os
import json
from dotenv import load_dotenv

load_dotenv()

router = APIRouter(prefix="/ai", tags=["ai"])

# Configuration IA
EMERGENT_LLM_KEY = os.getenv("EMERGENT_LLM_KEY")

def get_ai_chat(system_message: str):
    """Initialise un chat IA avec le système configuré"""
    try:
        from emergentintegrations.llm.chat import LlmChat, UserMessage
        chat = LlmChat(
            api_key=EMERGENT_LLM_KEY,
            session_id=f"erp-ai-{datetime.now().timestamp()}",
            system_message=system_message
        )
        # Utiliser GPT-4 (modèle stable et testé)
        chat.with_model("openai", "gpt-4")
        return chat
    except Exception as e:
        print(f"Erreur lors de l'initialisation du chat IA: {str(e)}")
        return None

async def safe_ai_call(chat, prompt: str):
    """Appel IA sécurisé avec gestion d'erreurs"""
    if chat is None:
        return {"error": "Chat IA non initialisé"}
    
    try:
        from emergentintegrations.llm.chat import UserMessage
        user_message = UserMessage(text=prompt)
        response = await chat.send_message(user_message)
        
        if response is None:
            return {"error": "Aucune réponse de l'IA"}
        
        # Convertir la réponse en string et nettoyer
        response_text = str(response).strip()
        
        # Essayer de parser le JSON
        try:
            if response_text.startswith("```json"):
                response_text = response_text.split("```json")[1].split("```")[0]
            elif response_text.startswith("```"):
                response_text = response_text.split("```")[1].split("```")[0]
            
            return json.loads(response_text)
        except json.JSONDecodeError:
            return {"raw_response": response_text, "note": "Réponse non structurée"}
            
    except Exception as e:
        return {"error": f"Erreur lors de l'appel IA: {str(e)}"}

# ==================== ATELIER - ANALYSE PRÉDICTIVE ====================

@router.post("/workshop/analyze-repair")
async def analyze_repair_order(
    repair_data: dict,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    """
    Analyse une réparation et fournit des recommandations IA
    """
    try:
        system_message = """Tu es un expert en mécanique automobile spécialisé dans l'analyse de réparations.
        Tu dois analyser les informations fournies et donner des recommandations précises et utiles.
        Réponds toujours en JSON avec les clés: estimated_cost, recommended_parts, estimated_time, potential_issues, recommendations."""
        
        chat = get_ai_chat(system_message)
        
        prompt = f"""
        Analyse cette réparation automobile:
        
        Véhicule: {repair_data.get('vehicle_make', '')} {repair_data.get('vehicle_model', '')} ({repair_data.get('vehicle_year', '')})
        Kilométrage: {repair_data.get('mileage', 0)} km
        Description du problème: {repair_data.get('description', '')}
        Services demandés: {', '.join(repair_data.get('services', []))}
        
        Fournis une analyse complète en JSON avec:
        - estimated_cost: Estimation du coût total en DA (fourchette min-max)
        - recommended_parts: Liste des pièces recommandées à vérifier/remplacer
        - estimated_time: Temps de réparation estimé en heures
        - potential_issues: Problèmes potentiels à anticiper
        - recommendations: Recommandations pour le client
        """
        
        analysis = await safe_ai_call(chat, prompt)
        
        return {
            "analysis": analysis,
            "timestamp": datetime.now(timezone.utc).isoformat()
        }
        
    except Exception as e:
        raise HTTPException(status_code=500, detail=f"Erreur lors de l'analyse IA: {str(e)}")

@router.post("/workshop/detect-anomalies")
async def detect_workshop_anomalies(
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    """
    Détecte les anomalies dans les ordres de réparation
    """
    try:
        # Récupérer les ordres de réparation récents
        repair_orders = await db.repair_orders.find({}).to_list(100)
        
        if not repair_orders:
            return {"anomalies": [], "message": "Aucune donnée à analyser"}
        
        # Préparer les données pour l'IA
        repair_summary = []
        for order in repair_orders[:20]:  # Limiter à 20 pour ne pas surcharger
            repair_summary.append({
                "repair_number": order.get("repair_number"),
                "total": order.get("total", 0),
                "status": order.get("status"),
                "created_at": order.get("created_at"),
                "completion_date": order.get("completion_date")
            })
        
        system_message = """Tu es un analyste de données expert en détection d'anomalies dans les opérations d'atelier.
        Identifie les patterns inhabituels, les outliers, et les potentiels problèmes.
        Réponds en JSON avec la clé 'anomalies' contenant une liste d'anomalies détectées."""
        
        chat = get_ai_chat(system_message)
        
        prompt = f"""
        Analyse ces ordres de réparation et détecte les anomalies potentielles:
        
        {json.dumps(repair_summary, indent=2)}
        
        Identifie:
        - Réparations avec des coûts inhabituellement élevés ou bas
        - Délais anormalement longs
        - Patterns suspects (ex: même client plusieurs fois en peu de temps)
        - Incohérences dans les données
        
        Réponds en JSON avec la structure:
        {{
            "anomalies": [
                {{
                    "repair_number": "REP-XXX",
                    "type": "cout_anormal|delai_long|pattern_suspect",
                    "severity": "low|medium|high",
                    "description": "Description détaillée",
                    "recommendation": "Action recommandée"
                }}
            ],
            "summary": "Résumé général"
        }}
        """
        
        result = await safe_ai_call(chat, prompt)
        
        # Assurer que la structure attendue est présente
        if "anomalies" not in result:
            if "error" in result:
                result = {"anomalies": [], **result}
            else:
                result = {"anomalies": [], "raw_response": result}
        
        return result
        
    except Exception as e:
        raise HTTPException(status_code=500, detail=f"Erreur lors de la détection d'anomalies: {str(e)}")

# ==================== FINANCE - ANALYSE FINANCIÈRE ====================

@router.post("/finance/analyze-transactions")
async def analyze_financial_transactions(
    period_days: int = 30,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    """
    Analyse les transactions financières et détecte les anomalies
    """
    try:
        # Récupérer les écritures comptables récentes
        start_date = (datetime.now(timezone.utc) - timedelta(days=period_days)).isoformat()
        journal_entries = await db.journal_entries.find({
            "date": {"$gte": start_date}
        }).to_list(100)
        
        if not journal_entries:
            return {"analysis": {"message": "Aucune transaction à analyser"}}
        
        # Préparer un résumé pour l'IA
        transactions_summary = []
        for entry in journal_entries[:30]:
            transactions_summary.append({
                "entry_number": entry.get("entry_number"),
                "date": entry.get("date"),
                "journal_type": entry.get("journal_type"),
                "total_debit": entry.get("total_debit", 0),
                "total_credit": entry.get("total_credit", 0),
                "description": entry.get("description", "")
            })
        
        system_message = """Tu es un expert comptable et analyste financier spécialisé en détection de fraude et optimisation financière.
        Analyse les transactions et identifie les anomalies, patterns suspects, et opportunités d'optimisation."""
        
        chat = get_ai_chat(system_message)
        
        prompt = f"""
        Analyse ces transactions financières des {period_days} derniers jours:
        
        {json.dumps(transactions_summary, indent=2)}
        
        Fournis une analyse complète en JSON:
        {{
            "anomalies": [
                {{"type": "...", "severity": "low|medium|high", "description": "...", "transaction": "..."}}
            ],
            "patterns": ["Pattern 1", "Pattern 2"],
            "recommendations": ["Recommandation 1", "Recommandation 2"],
            "risk_score": 0-100,
            "summary": "Résumé de l'analyse"
        }}
        """
        
        analysis = await safe_ai_call(chat, prompt)
        
        return {
            "analysis": analysis,
            "period_days": period_days,
            "transactions_analyzed": len(transactions_summary),
            "timestamp": datetime.now(timezone.utc).isoformat()
        }
        
    except Exception as e:
        raise HTTPException(status_code=500, detail=f"Erreur lors de l'analyse financière: {str(e)}")

@router.post("/finance/predict-cashflow")
async def predict_cashflow(
    months_ahead: int = 3,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    """
    Prédit les flux de trésorerie futurs basés sur l'historique
    """
    try:
        # Récupérer les transactions des 6 derniers mois
        start_date = (datetime.now(timezone.utc) - timedelta(days=180)).isoformat()
        journal_entries = await db.journal_entries.find({
            "date": {"$gte": start_date}
        }).to_list(500)
        
        if not journal_entries:
            return {"prediction": {"message": "Données insuffisantes pour prédiction"}}
        
        # Calculer les totaux mensuels
        monthly_data = {}
        for entry in journal_entries:
            date = datetime.fromisoformat(entry.get("date"))
            month_key = f"{date.year}-{date.month:02d}"
            if month_key not in monthly_data:
                monthly_data[month_key] = {"debit": 0, "credit": 0}
            monthly_data[month_key]["debit"] += entry.get("total_debit", 0)
            monthly_data[month_key]["credit"] += entry.get("total_credit", 0)
        
        system_message = """Tu es un analyste financier expert en prévisions de trésorerie.
        Analyse les données historiques et fournis des prédictions réalistes."""
        
        chat = get_ai_chat(system_message)
        
        prompt = f"""
        Données de trésorerie historiques (6 derniers mois):
        {json.dumps(monthly_data, indent=2)}
        
        Prédit les flux de trésorerie pour les {months_ahead} prochains mois.
        
        Réponds en JSON:
        {{
            "predictions": [
                {{"month": "2025-04", "predicted_inflow": 0, "predicted_outflow": 0, "net_cashflow": 0, "confidence": "low|medium|high"}}
            ],
            "trends": ["Tendance 1", "Tendance 2"],
            "risks": ["Risque 1", "Risque 2"],
            "recommendations": ["Recommandation 1"],
            "summary": "Résumé des prédictions"
        }}
        """
        
        prediction = await safe_ai_call(chat, prompt)
        
        return {
            "prediction": prediction,
            "months_ahead": months_ahead,
            "historical_months": len(monthly_data),
            "timestamp": datetime.now(timezone.utc).isoformat()
        }
        
    except Exception as e:
        raise HTTPException(status_code=500, detail=f"Erreur lors de la prédiction: {str(e)}")

# ==================== RH - ANALYSE DES EMPLOYÉS ====================

@router.post("/hr/analyze-absences")
async def analyze_employee_absences(
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    """
    Analyse les patterns d'absences et congés des employés
    """
    try:
        # Récupérer les employés et leurs demandes de congés
        employees = await db.employees.find({"status": "active"}).to_list(100)
        leave_requests = await db.leave_requests.find({}).to_list(500)
        
        if not employees or not leave_requests:
            return {"analysis": {"message": "Données insuffisantes"}}
        
        # Préparer les données
        absence_data = {}
        for emp in employees:
            absence_data[emp.get("employee_number")] = {
                "name": f"{emp.get('first_name')} {emp.get('last_name')}",
                "department": emp.get("department"),
                "position": emp.get("position"),
                "leaves": []
            }
        
        for leave in leave_requests:
            emp_id = leave.get("employee_id")
            if emp_id in absence_data:
                absence_data[emp_id]["leaves"].append({
                    "type": leave.get("leave_type"),
                    "start": leave.get("start_date"),
                    "end": leave.get("end_date"),
                    "days": leave.get("days_count"),
                    "status": leave.get("status")
                })
        
        system_message = """Tu es un expert RH spécialisé en analyse des absences et gestion du personnel.
        Identifie les patterns, prédis les tendances, et recommande des actions."""
        
        chat = get_ai_chat(system_message)
        
        prompt = f"""
        Analyse les données d'absences des employés:
        
        {json.dumps(list(absence_data.values())[:20], indent=2)}
        
        Fournis une analyse complète en JSON:
        {{
            "patterns": ["Pattern détecté 1", "Pattern détecté 2"],
            "anomalies": [
                {{"employee": "...", "issue": "...", "severity": "low|medium|high"}}
            ],
            "predictions": [
                {{"period": "...", "expected_absences": 0, "departments_affected": []}}
            ],
            "recommendations": ["Recommandation 1", "Recommandation 2"],
            "risk_assessment": {{
                "understaffing_risk": "low|medium|high",
                "departments_at_risk": []
            }},
            "summary": "Résumé de l'analyse"
        }}
        """
        
        analysis = await safe_ai_call(chat, prompt)
        
        return {
            "analysis": analysis,
            "employees_analyzed": len(employees),
            "leave_requests_analyzed": len(leave_requests),
            "timestamp": datetime.now(timezone.utc).isoformat()
        }
        
    except Exception as e:
        raise HTTPException(status_code=500, detail=f"Erreur lors de l'analyse RH: {str(e)}")

# ==================== DASHBOARD GÉNÉRAL - INSIGHTS IA ====================

@router.get("/insights/general")
async def get_general_insights(
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    """
    Génère des insights IA généraux sur l'ensemble de l'entreprise
    """
    try:
        # Récupérer les données clés de chaque module
        repair_orders_count = await db.repair_orders.count_documents({})
        active_employees = await db.employees.count_documents({"status": "active"})
        recent_transactions = await db.journal_entries.count_documents({
            "date": {"$gte": (datetime.now(timezone.utc) - timedelta(days=30)).isoformat()}
        })
        pending_leaves = await db.leave_requests.count_documents({"status": "pending"})
        
        # Récupérer quelques données récentes pour contexte
        recent_repairs = await db.repair_orders.find({}).sort("created_at", -1).limit(5).to_list(5)
        recent_journal = await db.journal_entries.find({}).sort("date", -1).limit(5).to_list(5)
        
        business_data = {
            "kpis": {
                "repair_orders_total": repair_orders_count,
                "active_employees": active_employees,
                "monthly_transactions": recent_transactions,
                "pending_leave_requests": pending_leaves
            },
            "recent_activity": {
                "repairs": [
                    {
                        "repair_number": r.get("repair_number"),
                        "total": r.get("total", 0),
                        "status": r.get("status")
                    } for r in recent_repairs
                ],
                "transactions": [
                    {
                        "entry_number": t.get("entry_number"),
                        "total_debit": t.get("total_debit", 0),
                        "total_credit": t.get("total_credit", 0)
                    } for t in recent_journal
                ]
            }
        }
        
        system_message = """Tu es un consultant d'entreprise expert qui fournit des insights stratégiques basés sur les données.
        Analyse la situation globale et fournis des recommandations actionnables."""
        
        chat = get_ai_chat(system_message)
        
        prompt = f"""
        Analyse les données globales de cette entreprise de pièces automobiles:
        
        {json.dumps(business_data, indent=2)}
        
        Fournis des insights en JSON:
        {{
            "health_score": 0-100,
            "health_status": "excellent|good|moderate|poor",
            "key_metrics": [
                {{"metric": "...", "value": "...", "trend": "up|down|stable", "comment": "..."}}
            ],
            "alerts": [
                {{"priority": "high|medium|low", "message": "...", "action": "..."}}
            ],
            "opportunities": ["Opportunité 1", "Opportunité 2"],
            "recommendations": ["Recommandation prioritaire 1", "Recommandation 2"],
            "executive_summary": "Résumé pour la direction (2-3 phrases)"
        }}
        """
        
        insights = await safe_ai_call(chat, prompt)
        
        return {
            "insights": insights,
            "data_snapshot": business_data["kpis"],
            "timestamp": datetime.now(timezone.utc).isoformat()
        }
        
    except Exception as e:
        raise HTTPException(status_code=500, detail=f"Erreur lors de la génération d'insights: {str(e)}")