from pydantic import BaseModel, Field, EmailStr, ConfigDict
from typing import Optional, List, Dict
from datetime import datetime, timezone
from enum import Enum
import uuid

# Enums
class UserRole(str, Enum):
    SUPERADMIN = "superadmin"
    ADMIN = "admin"
    MANAGER = "manager"
    CLIENT = "client"

class ClientCategory(str, Enum):
    PUBLIC = "public"
    GARAGE = "garage"
    SEMI_WHOLESALE = "semi_wholesale"
    WHOLESALE = "wholesale"
    VIP = "vip"

class OrderStatus(str, Enum):
    PENDING = "pending"
    CONFIRMED = "confirmed"
    PREPARING = "preparing"
    READY = "ready"
    IN_DELIVERY = "in_delivery"
    DELIVERED = "delivered"
    CANCELLED = "cancelled"

class RepairOrderStatus(str, Enum):
    DRAFT = "draft"
    SCHEDULED = "scheduled"
    IN_PROGRESS = "in_progress"
    WAITING_PARTS = "waiting_parts"
    COMPLETED = "completed"
    INVOICED = "invoiced"
    PAID = "paid"
    CANCELLED = "cancelled"

class RepairOrderPriority(str, Enum):
    LOW = "low"
    MEDIUM = "medium"
    HIGH = "high"
    URGENT = "urgent"

# Base Models
class BaseDBModel(BaseModel):
    model_config = ConfigDict(extra="ignore")
    id: str = Field(default_factory=lambda: str(uuid.uuid4()))
    created_at: datetime = Field(default_factory=lambda: datetime.now(timezone.utc))
    updated_at: Optional[datetime] = None

# Permissions System
class ModulePermission(BaseModel):
    """Permissions pour un module spécifique"""
    view: bool = False      # Consulter/Voir
    create: bool = False    # Créer
    edit: bool = False      # Modifier
    delete: bool = False    # Supprimer
    approve: bool = False   # Approuver (pour congés, commandes, etc.)

class UserPermissions(BaseModel):
    """Permissions granulaires par module"""
    dashboard: ModulePermission = ModulePermission()
    products: ModulePermission = ModulePermission()
    stock: ModulePermission = ModulePermission()
    orders: ModulePermission = ModulePermission()
    clients: ModulePermission = ModulePermission()
    workshop: ModulePermission = ModulePermission()
    finance: ModulePermission = ModulePermission()
    fiscal: ModulePermission = ModulePermission()
    hr: ModulePermission = ModulePermission()
    ged: ModulePermission = ModulePermission()
    reports: ModulePermission = ModulePermission()
    users: ModulePermission = ModulePermission()
    imports: ModulePermission = ModulePermission()
    categories: ModulePermission = ModulePermission()
    brands: ModulePermission = ModulePermission()

# User Models
class User(BaseDBModel):
    email: EmailStr
    password_hash: str
    full_name: str
    phone: Optional[str] = None
    role: UserRole = UserRole.CLIENT
    is_active: bool = True
    language: str = "fr"  # fr, ar, en
    permissions: Optional[UserPermissions] = None  # Permissions granulaires (si None, utiliser les permissions par défaut du rôle)

class UserCreate(BaseModel):
    email: EmailStr
    password: str
    full_name: str
    phone: Optional[str] = None
    role: UserRole = UserRole.CLIENT
    language: str = "fr"
    permissions: Optional[UserPermissions] = None

class UserLogin(BaseModel):
    email: EmailStr
    password: str

class UserResponse(BaseModel):
    id: str
    email: EmailStr
    full_name: str
    phone: Optional[str] = None
    role: UserRole
    is_active: bool
    language: str
    created_at: datetime
    permissions: Optional[UserPermissions] = None

# Brand Models
class Brand(BaseDBModel):
    name: str
    logo: Optional[str] = None
    country: Optional[str] = None
    description: Optional[str] = None

class BrandCreate(BaseModel):
    name: str
    logo: Optional[str] = None
    country: Optional[str] = None
    description: Optional[str] = None

# Category Models
class Category(BaseDBModel):
    name: str
    description: Optional[str] = None
    parent_id: Optional[str] = None
    icon: Optional[str] = None

class CategoryCreate(BaseModel):
    name: str
    description: Optional[str] = None
    parent_id: Optional[str] = None
    icon: Optional[str] = None

# Product Models
class ProductPrices(BaseModel):
    public: float
    garage: float
    semi_wholesale: float
    wholesale: float

class Product(BaseDBModel):
    reference: str
    name: str
    description: Optional[str] = None
    category_id: str
    brand_id: str
    stock_qty: int = 0
    min_stock: int = 10
    prices: ProductPrices
    images: List[str] = []
    specifications: Dict[str, str] = {}
    is_active: bool = True
    oem_references: List[str] = []  # Références OEM équivalentes

class ProductCreate(BaseModel):
    reference: str
    name: str
    description: Optional[str] = None
    category_id: str
    brand_id: str
    stock_qty: int = 0
    min_stock: int = 10
    prices: ProductPrices
    images: List[str] = []
    specifications: Dict[str, str] = {}
    oem_references: List[str] = []

class ProductUpdate(BaseModel):
    reference: Optional[str] = None
    name: Optional[str] = None
    description: Optional[str] = None
    category_id: Optional[str] = None
    brand_id: Optional[str] = None
    stock_qty: Optional[int] = None
    min_stock: Optional[int] = None
    prices: Optional[ProductPrices] = None
    images: Optional[List[str]] = None
    specifications: Optional[Dict[str, str]] = None
    is_active: Optional[bool] = None
    oem_references: Optional[List[str]] = None

# Client Models
class Client(BaseDBModel):
    user_id: str
    company_name: str
    category: ClientCategory = ClientCategory.PUBLIC
    credit_limit: float = 0.0
    payment_terms: int = 0  # days
    address: Optional[str] = None
    city: Optional[str] = None
    tax_id: Optional[str] = None  # NIF
    commercial_register: Optional[str] = None  # RC
    balance: float = 0.0

class ClientCreate(BaseModel):
    user_id: str
    company_name: str
    category: ClientCategory = ClientCategory.PUBLIC
    credit_limit: float = 0.0
    payment_terms: int = 0
    address: Optional[str] = None
    city: Optional[str] = None
    tax_id: Optional[str] = None
    commercial_register: Optional[str] = None

class ClientUpdate(BaseModel):
    company_name: Optional[str] = None
    category: Optional[ClientCategory] = None
    credit_limit: Optional[float] = None
    payment_terms: Optional[int] = None
    address: Optional[str] = None
    city: Optional[str] = None
    tax_id: Optional[str] = None
    commercial_register: Optional[str] = None

# Order Models
class OrderItem(BaseModel):
    product_id: str
    product_name: str
    product_reference: str
    quantity: int
    unit_price: float
    subtotal: float

class Order(BaseDBModel):
    order_number: str
    client_id: str
    client_name: str
    status: OrderStatus = OrderStatus.PENDING
    items: List[OrderItem]
    subtotal: float
    tax_amount: float  # TVA 19%
    total: float
    notes: Optional[str] = None
    delivery_address: Optional[str] = None

class OrderCreate(BaseModel):
    client_id: str
    items: List[OrderItem]
    notes: Optional[str] = None
    delivery_address: Optional[str] = None

class OrderStatusUpdate(BaseModel):
    status: OrderStatus

# Dashboard Stats
class DashboardStats(BaseModel):
    total_products: int
    low_stock_products: int
    total_clients: int
    pending_orders: int
    total_revenue: float
    recent_orders: List[Dict]

# Workshop/Repair Order Models
class RepairServiceItem(BaseModel):
    description: str
    labor_hours: float
    labor_rate: float
    labor_cost: float

class RepairPartItem(BaseModel):
    product_id: str
    product_name: str
    product_reference: str
    quantity: int
    unit_price: float
    subtotal: float

class RepairOrder(BaseDBModel):
    repair_number: str
    client_id: Optional[str] = None  # Can be null for walk-in customers
    client_name: str
    vehicle_make: str
    vehicle_model: str
    vehicle_year: Optional[int] = None
    vehicle_vin: Optional[str] = None
    vehicle_plate: str
    mileage: Optional[int] = None
    status: RepairOrderStatus = RepairOrderStatus.DRAFT
    priority: RepairOrderPriority = RepairOrderPriority.MEDIUM
    scheduled_date: Optional[datetime] = None
    completion_date: Optional[datetime] = None
    assigned_technician: Optional[str] = None
    description: str
    diagnosis: Optional[str] = None
    services: List[RepairServiceItem] = []
    parts: List[RepairPartItem] = []
    labor_subtotal: float = 0.0
    parts_subtotal: float = 0.0
    subtotal: float = 0.0
    tax_amount: float = 0.0  # TVA 19%
    total: float = 0.0
    notes: Optional[str] = None
    invoice_id: Optional[str] = None

class RepairOrderCreate(BaseModel):
    client_id: Optional[str] = None
    client_name: str
    vehicle_make: str
    vehicle_model: str
    vehicle_year: Optional[int] = None
    vehicle_vin: Optional[str] = None
    vehicle_plate: str
    mileage: Optional[int] = None
    priority: RepairOrderPriority = RepairOrderPriority.MEDIUM
    scheduled_date: Optional[datetime] = None
    assigned_technician: Optional[str] = None
    description: str
    diagnosis: Optional[str] = None
    services: List[RepairServiceItem] = []
    parts: List[RepairPartItem] = []
    notes: Optional[str] = None

class RepairOrderUpdate(BaseModel):
    client_id: Optional[str] = None
    client_name: Optional[str] = None
    vehicle_make: Optional[str] = None
    vehicle_model: Optional[str] = None
    vehicle_year: Optional[int] = None
    vehicle_vin: Optional[str] = None
    vehicle_plate: Optional[str] = None
    mileage: Optional[int] = None
    status: Optional[RepairOrderStatus] = None
    priority: Optional[RepairOrderPriority] = None
    scheduled_date: Optional[datetime] = None
    completion_date: Optional[datetime] = None
    assigned_technician: Optional[str] = None
    description: Optional[str] = None
    diagnosis: Optional[str] = None
    services: Optional[List[RepairServiceItem]] = None
    parts: Optional[List[RepairPartItem]] = None
    notes: Optional[str] = None

class RepairOrderStatusUpdate(BaseModel):
    status: RepairOrderStatus


# ==================== FINANCE & COMPTABILITÉ MODELS ====================

class AccountType(str, Enum):
    ASSET = "asset"  # Actif
    LIABILITY = "liability"  # Passif
    EQUITY = "equity"  # Capitaux propres
    REVENUE = "revenue"  # Produits
    EXPENSE = "expense"  # Charges

class JournalType(str, Enum):
    SALES = "sales"  # Journal ventes
    PURCHASES = "purchases"  # Journal achats
    BANK = "bank"  # Journal banque
    CASH = "cash"  # Journal caisse
    OPERATIONS = "operations"  # Journal opérations diverses

class PaymentMethod(str, Enum):
    CASH = "cash"
    CHECK = "check"
    TRANSFER = "transfer"
    CARD = "card"
    LCN = "lcn"  # Lettre de change normalisée
    TRAITE = "traite"

class TransactionStatus(str, Enum):
    DRAFT = "draft"
    POSTED = "posted"
    VALIDATED = "validated"
    CANCELLED = "cancelled"

# Plan Comptable (Chart of Accounts)
class Account(BaseDBModel):
    code: str  # Ex: 411001
    name: str
    account_type: AccountType
    parent_code: Optional[str] = None
    is_active: bool = True
    balance: float = 0.0

class AccountCreate(BaseModel):
    code: str
    name: str
    account_type: AccountType
    parent_code: Optional[str] = None

# Écriture Comptable (Journal Entry)
class JournalEntryLine(BaseModel):
    account_code: str
    account_name: str
    debit: float = 0.0
    credit: float = 0.0
    description: Optional[str] = None

class JournalEntry(BaseDBModel):
    entry_number: str
    journal_type: JournalType
    date: datetime
    description: str
    reference: Optional[str] = None  # Facture, BL, etc.
    lines: List[JournalEntryLine]
    total_debit: float
    total_credit: float
    status: TransactionStatus = TransactionStatus.DRAFT
    posted_by: Optional[str] = None
    validated_by: Optional[str] = None

class JournalEntryCreate(BaseModel):
    journal_type: JournalType
    date: datetime
    description: str
    reference: Optional[str] = None
    lines: List[JournalEntryLine]

# Banque
class BankAccount(BaseDBModel):
    bank_name: str
    account_number: str
    rib: str  # Relevé Identité Bancaire
    iban: Optional[str] = None
    swift: Optional[str] = None
    balance: float = 0.0
    currency: str = "DZD"
    is_active: bool = True

class BankTransaction(BaseDBModel):
    bank_account_id: str
    date: datetime
    description: str
    reference: Optional[str] = None
    debit: float = 0.0
    credit: float = 0.0
    balance: float
    reconciled: bool = False
    journal_entry_id: Optional[str] = None

# Caisse
class CashRegister(BaseDBModel):
    name: str  # Caisse principale, Caisse atelier, etc.
    site_id: Optional[str] = None
    responsible: str
    balance: float = 0.0
    is_active: bool = True

class CashTransaction(BaseDBModel):
    cash_register_id: str
    date: datetime
    description: str
    reference: Optional[str] = None
    debit: float = 0.0
    credit: float = 0.0
    balance: float
    payment_method: PaymentMethod
    journal_entry_id: Optional[str] = None

# Encaissement Client
class CustomerPayment(BaseDBModel):
    customer_id: str
    customer_name: str
    amount: float
    payment_date: datetime
    payment_method: PaymentMethod
    reference: Optional[str] = None  # N° chèque, virement, etc.
    bank_account_id: Optional[str] = None
    cash_register_id: Optional[str] = None
    invoice_ids: List[str] = []  # Factures payées
    notes: Optional[str] = None
    journal_entry_id: Optional[str] = None

class SupplierPayment(BaseDBModel):
    supplier_id: str
    supplier_name: str
    amount: float
    payment_date: datetime
    payment_method: PaymentMethod
    reference: Optional[str] = None
    bank_account_id: Optional[str] = None
    invoice_ids: List[str] = []
    notes: Optional[str] = None
    journal_entry_id: Optional[str] = None

# Budget
class Budget(BaseDBModel):
    name: str
    fiscal_year: int
    start_date: datetime
    end_date: datetime
    account_code: str
    budgeted_amount: float
    spent_amount: float = 0.0
    variance: float = 0.0

# Immobilisations
class AssetCategory(str, Enum):
    BUILDING = "building"
    VEHICLE = "vehicle"
    EQUIPMENT = "equipment"
    FURNITURE = "furniture"
    IT = "it"
    OTHER = "other"

class Asset(BaseDBModel):
    name: str
    category: AssetCategory
    purchase_date: datetime
    purchase_value: float
    depreciation_method: str = "linear"  # linear, declining
    depreciation_rate: float  # Percentage
    useful_life_years: int
    residual_value: float = 0.0
    accumulated_depreciation: float = 0.0
    net_book_value: float
    location: Optional[str] = None
    serial_number: Optional[str] = None
    is_active: bool = True

# ==================== FISCALITÉ ALGÉRIENNE MODELS ====================

class TaxType(str, Enum):
    TVA = "tva"  # TVA 19%
    IBS = "ibs"  # Impôt sur Bénéfices Sociétés
    TAP = "tap"  # Taxe Activité Professionnelle
    IRG = "irg"  # Impôt Revenu Global
    CACOBATPH = "cacobatph"
    FORMATION = "formation"
    APPRENTISSAGE = "apprentissage"

class TaxDeclarationStatus(str, Enum):
    DRAFT = "draft"
    SUBMITTED = "submitted"
    PAID = "paid"
    LATE = "late"

# Déclaration TVA (G50)
class TVADeclaration(BaseDBModel):
    period_month: int
    period_year: int
    declaration_date: datetime
    due_date: datetime
    
    # TVA Collectée
    sales_ht: float = 0.0
    tva_collected: float = 0.0
    
    # TVA Récupérable
    purchases_ht: float = 0.0
    tva_recoverable: float = 0.0
    
    # TVA à payer ou crédit
    tva_due: float = 0.0  # Si positif = à payer
    tva_credit: float = 0.0  # Si négatif = crédit
    
    status: TaxDeclarationStatus = TaxDeclarationStatus.DRAFT
    submitted_by: Optional[str] = None
    payment_reference: Optional[str] = None

# Déclaration IBS
class IBSDeclaration(BaseDBModel):
    fiscal_year: int
    declaration_type: str  # "acompte" ou "annual"
    quarter: Optional[int] = None  # 1, 2, 3, 4 pour acomptes
    declaration_date: datetime
    due_date: datetime
    
    # Résultat fiscal
    accounting_result: float = 0.0
    reintegrations: float = 0.0
    deductions: float = 0.0
    taxable_result: float = 0.0
    
    # IBS
    ibs_rate: float = 26.0  # 26% en Algérie
    ibs_amount: float = 0.0
    
    status: TaxDeclarationStatus = TaxDeclarationStatus.DRAFT
    payment_reference: Optional[str] = None

# Déclaration TAP
class TAPDeclaration(BaseDBModel):
    period_month: int
    period_year: int
    declaration_date: datetime
    due_date: datetime
    
    turnover_ht: float = 0.0
    tap_rate: float = 2.0  # 2% du CA HT
    tap_amount: float = 0.0
    
    status: TaxDeclarationStatus = TaxDeclarationStatus.DRAFT
    payment_reference: Optional[str] = None

# Calendrier Fiscal
class FiscalDeadline(BaseDBModel):
    tax_type: TaxType
    name: str
    due_date: datetime
    period_month: Optional[int] = None
    period_year: int
    is_completed: bool = False
    declaration_id: Optional[str] = None
    notes: Optional[str] = None

# ==================== RH & PAIE MODELS ====================

class EmployeeStatus(str, Enum):
    ACTIVE = "active"
    ON_LEAVE = "on_leave"
    SUSPENDED = "suspended"
    TERMINATED = "terminated"

class ContractType(str, Enum):
    CDI = "cdi"  # Contrat Durée Indéterminée
    CDD = "cdd"  # Contrat Durée Déterminée
    DAIP = "daip"  # Contrat Pré-emploi
    INTERNSHIP = "internship"

class LeaveType(str, Enum):
    ANNUAL = "annual"  # Congés annuels
    SICK = "sick"  # Maladie
    MATERNITY = "maternity"  # Maternité
    PATERNITY = "paternity"  # Paternité
    UNPAID = "unpaid"  # Sans solde
    EXCEPTIONAL = "exceptional"  # Exceptionnel

class AbsenceStatus(str, Enum):
    PENDING = "pending"
    APPROVED = "approved"
    REJECTED = "rejected"
    CANCELLED = "cancelled"

# Employé
class Employee(BaseDBModel):
    # Informations personnelles
    first_name: str
    last_name: str
    date_of_birth: datetime
    place_of_birth: str
    gender: str  # M/F
    nationality: str = "Algérienne"
    
    # Documents
    cin_number: str  # Carte Identité Nationale
    cin_issue_date: Optional[datetime] = None
    cin_expiry_date: Optional[datetime] = None
    social_security_number: str
    
    # Contact
    email: Optional[str] = None
    phone: str
    address: str
    city: str
    postal_code: Optional[str] = None
    
    # Emploi
    employee_number: str  # Matricule
    hire_date: datetime
    contract_type: ContractType
    contract_end_date: Optional[datetime] = None
    position: str  # Poste
    department: str  # Service
    site_id: Optional[str] = None
    
    # Rémunération
    base_salary: float
    bank_account: Optional[str] = None
    
    # Statut
    status: EmployeeStatus = EmployeeStatus.ACTIVE
    termination_date: Optional[datetime] = None
    termination_reason: Optional[str] = None
    
    # Divers
    photo_url: Optional[str] = None
    notes: Optional[str] = None

class EmployeeCreate(BaseModel):
    first_name: str
    last_name: str
    date_of_birth: datetime
    place_of_birth: str
    gender: str  # M/F
    cin_number: str
    social_security_number: str
    phone: str
    address: str
    city: str
    employee_number: str
    hire_date: datetime
    contract_type: ContractType
    position: str
    department: str
    base_salary: float

# Pointage
class Attendance(BaseDBModel):
    employee_id: str
    date: datetime
    check_in: Optional[datetime] = None
    check_out: Optional[datetime] = None
    worked_hours: float = 0.0
    overtime_hours: float = 0.0
    notes: Optional[str] = None

# Congés/Absences
class LeaveRequest(BaseDBModel):
    employee_id: str
    employee_name: str
    leave_type: LeaveType
    start_date: datetime
    end_date: datetime
    days_count: int
    reason: Optional[str] = None
    status: AbsenceStatus = AbsenceStatus.PENDING
    approved_by: Optional[str] = None
    approval_date: Optional[datetime] = None
    rejection_reason: Optional[str] = None

# Bulletin de Paie
class PayslipItem(BaseModel):
    description: str
    amount: float
    is_taxable: bool = True

class Payslip(BaseDBModel):
    employee_id: str
    employee_name: str
    employee_number: str
    period_month: int
    period_year: int
    
    # Salaire
    base_salary: float
    allowances: List[PayslipItem] = []  # Primes, indemnités
    gross_salary: float
    
    # Charges sociales
    cnss_employee: float  # 9%
    cnss_employer: float  # 26%
    
    # IRG
    taxable_salary: float
    irg_amount: float
    
    # Net
    total_deductions: float
    net_salary: float
    
    # Dates
    payment_date: datetime
    
    # Statut
    is_paid: bool = False
    payment_reference: Optional[str] = None

# Formation
class Training(BaseDBModel):
    title: str
    description: Optional[str] = None
    provider: Optional[str] = None
    start_date: datetime
    end_date: datetime
    duration_hours: int
    cost: float = 0.0
    location: Optional[str] = None
    max_participants: Optional[int] = None
    employee_ids: List[str] = []

# ==================== GED (GESTION ÉLECTRONIQUE DOCUMENTS) MODELS ====================

class DocumentCategory(str, Enum):
    COMMERCIAL = "commercial"  # Devis, factures, BL
    PURCHASE = "purchase"  # Commandes, factures fournisseurs
    HR = "hr"  # Contrats, fiches paie
    ADMINISTRATIVE = "administrative"  # RC, NIS, NIF
    CUSTOMS = "customs"  # D10, D48, manifestes
    QUALITY = "quality"  # Certifications ISO
    LEGAL = "legal"  # Contrats, PV
    WORKSHOP = "workshop"  # OR, photos véhicules

class DocumentStatus(str, Enum):
    DRAFT = "draft"
    PENDING_APPROVAL = "pending_approval"
    APPROVED = "approved"
    REJECTED = "rejected"
    ARCHIVED = "archived"

class Document(BaseDBModel):
    name: str
    category: DocumentCategory
    description: Optional[str] = None
    file_path: str
    file_type: str  # pdf, docx, jpg, etc.
    file_size: int  # bytes
    
    # Métadonnées
    tags: List[str] = []
    reference: Optional[str] = None  # N° facture, contrat, etc.
    date: Optional[datetime] = None
    
    # Workflow
    status: DocumentStatus = DocumentStatus.DRAFT
    uploaded_by: str
    version: int = 1
    parent_document_id: Optional[str] = None  # For versioning
    
    # Validation
    requires_approval: bool = False
    approved_by: Optional[str] = None
    approval_date: Optional[datetime] = None
    rejection_reason: Optional[str] = None
    
    # Relations
    linked_entity_type: Optional[str] = None  # order, invoice, employee, etc.
    linked_entity_id: Optional[str] = None
    
    # Archivage
    retention_years: Optional[int] = None
    archive_date: Optional[datetime] = None

class DocumentCreate(BaseModel):
    name: str
    category: DocumentCategory
    description: Optional[str] = None
    file_path: str
    file_type: str
    file_size: int
    tags: List[str] = []
    reference: Optional[str] = None
    requires_approval: bool = False

class DocumentApproval(BaseModel):
    approved: bool
    rejection_reason: Optional[str] = None
