from fastapi import Header, HTTPException, status
from typing import Optional
from auth import decode_token
from models import UserRole
from motor.motor_asyncio import AsyncIOMotorClient
import os

# MongoDB connection for dependency injection
_mongo_client = None
_db = None

def get_mongo_client():
    global _mongo_client
    if _mongo_client is None:
        mongo_url = os.environ.get('MONGO_URL')
        _mongo_client = AsyncIOMotorClient(mongo_url)
    return _mongo_client

def get_db():
    """Dependency to get database instance"""
    global _db
    if _db is None:
        client = get_mongo_client()
        _db = client[os.environ.get('DB_NAME', 'erp_auto')]
    return _db

async def get_current_user(authorization: Optional[str] = Header(None)):
    if not authorization:
        raise HTTPException(
            status_code=status.HTTP_401_UNAUTHORIZED,
            detail="Not authenticated"
        )
    
    try:
        scheme, token = authorization.split()
        if scheme.lower() != "bearer":
            raise HTTPException(
                status_code=status.HTTP_401_UNAUTHORIZED,
                detail="Invalid authentication scheme"
            )
    except ValueError:
        raise HTTPException(
            status_code=status.HTTP_401_UNAUTHORIZED,
            detail="Invalid authorization header"
        )
    
    payload = decode_token(token)
    if not payload:
        raise HTTPException(
            status_code=status.HTTP_401_UNAUTHORIZED,
            detail="Invalid or expired token"
        )
    
    return payload

async def require_role(roles: list[UserRole]):
    async def role_checker(current_user: dict = None):
        if not current_user:
            raise HTTPException(
                status_code=status.HTTP_401_UNAUTHORIZED,
                detail="Not authenticated"
            )
        
        user_role = current_user.get("role")
        if user_role not in [role.value for role in roles]:
            raise HTTPException(
                status_code=status.HTTP_403_FORBIDDEN,
                detail="Insufficient permissions"
            )
        return current_user
    return role_checker
